/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2023      Michael R. Crusoe <crusoe@debian.org>
 */

#define SIMDE_TEST_X86_AVX512_INSN reduce

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/reduce.h>

#if defined(SIMDE_FLOAT16_IS_SCALAR)
static int
test_simde_mm512_reduce_max_ph (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float16 a[32];
    const simde_float16 r;
  } test_vec[] = {
    { { SIMDE_FLOAT16_VALUE(  -185.88), SIMDE_FLOAT16_VALUE(  -986.50), SIMDE_FLOAT16_VALUE(     2.45), SIMDE_FLOAT16_VALUE(   247.38),
        SIMDE_FLOAT16_VALUE(  -727.00), SIMDE_FLOAT16_VALUE(  -893.50), SIMDE_FLOAT16_VALUE(   875.50), SIMDE_FLOAT16_VALUE(   709.00),
        SIMDE_FLOAT16_VALUE(   406.75), SIMDE_FLOAT16_VALUE(  -130.75), SIMDE_FLOAT16_VALUE(  -631.00), SIMDE_FLOAT16_VALUE(   933.50),
        SIMDE_FLOAT16_VALUE(    90.00), SIMDE_FLOAT16_VALUE(  -937.00), SIMDE_FLOAT16_VALUE(  -754.00), SIMDE_FLOAT16_VALUE(    60.22),
        SIMDE_FLOAT16_VALUE(   233.00), SIMDE_FLOAT16_VALUE(  -323.25), SIMDE_FLOAT16_VALUE(    83.06), SIMDE_FLOAT16_VALUE(  -224.38),
        SIMDE_FLOAT16_VALUE(   860.50), SIMDE_FLOAT16_VALUE(  -115.75), SIMDE_FLOAT16_VALUE(   962.00), SIMDE_FLOAT16_VALUE(  -706.50),
        SIMDE_FLOAT16_VALUE(   742.00), SIMDE_FLOAT16_VALUE(  -185.25), SIMDE_FLOAT16_VALUE(  -100.25), SIMDE_FLOAT16_VALUE(  -953.00),
        SIMDE_FLOAT16_VALUE(   934.50), SIMDE_FLOAT16_VALUE(  -124.06), SIMDE_FLOAT16_VALUE(   835.00), SIMDE_FLOAT16_VALUE(  -251.38) },
      SIMDE_FLOAT16_VALUE(   962.00) },
    { { SIMDE_FLOAT16_VALUE(  -110.44), SIMDE_FLOAT16_VALUE(  -162.62), SIMDE_FLOAT16_VALUE(   996.00), SIMDE_FLOAT16_VALUE(   162.62),
        SIMDE_FLOAT16_VALUE(   -56.28), SIMDE_FLOAT16_VALUE(   871.50), SIMDE_FLOAT16_VALUE(  -128.38), SIMDE_FLOAT16_VALUE(  -649.50),
        SIMDE_FLOAT16_VALUE(  -259.25), SIMDE_FLOAT16_VALUE(   240.50), SIMDE_FLOAT16_VALUE(  -716.50), SIMDE_FLOAT16_VALUE(   830.50),
        SIMDE_FLOAT16_VALUE(   303.50), SIMDE_FLOAT16_VALUE(  -470.00), SIMDE_FLOAT16_VALUE(  -109.06), SIMDE_FLOAT16_VALUE(  -463.50),
        SIMDE_FLOAT16_VALUE(   206.75), SIMDE_FLOAT16_VALUE(   974.00), SIMDE_FLOAT16_VALUE(   312.25), SIMDE_FLOAT16_VALUE(    67.25),
        SIMDE_FLOAT16_VALUE(  -141.75), SIMDE_FLOAT16_VALUE(   274.25), SIMDE_FLOAT16_VALUE(   360.75), SIMDE_FLOAT16_VALUE(  -400.00),
        SIMDE_FLOAT16_VALUE(  -911.00), SIMDE_FLOAT16_VALUE(  -739.50), SIMDE_FLOAT16_VALUE(  -353.00), SIMDE_FLOAT16_VALUE(  -976.50),
        SIMDE_FLOAT16_VALUE(   136.38), SIMDE_FLOAT16_VALUE(  -518.00), SIMDE_FLOAT16_VALUE(  -227.88), SIMDE_FLOAT16_VALUE(  -974.00) },
      SIMDE_FLOAT16_VALUE(   996.00) },
    { { SIMDE_FLOAT16_VALUE(   319.25), SIMDE_FLOAT16_VALUE(  -232.00), SIMDE_FLOAT16_VALUE(   188.62), SIMDE_FLOAT16_VALUE(  -737.00),
        SIMDE_FLOAT16_VALUE(  -360.50), SIMDE_FLOAT16_VALUE(  -940.00), SIMDE_FLOAT16_VALUE(  -386.50), SIMDE_FLOAT16_VALUE(   380.25),
        SIMDE_FLOAT16_VALUE(   300.75), SIMDE_FLOAT16_VALUE(  -102.81), SIMDE_FLOAT16_VALUE(   211.00), SIMDE_FLOAT16_VALUE(  -395.50),
        SIMDE_FLOAT16_VALUE(   427.00), SIMDE_FLOAT16_VALUE(  -898.00), SIMDE_FLOAT16_VALUE(   141.00), SIMDE_FLOAT16_VALUE(  -366.25),
        SIMDE_FLOAT16_VALUE(  -924.00), SIMDE_FLOAT16_VALUE(  -546.50), SIMDE_FLOAT16_VALUE(   701.00), SIMDE_FLOAT16_VALUE(   -65.81),
        SIMDE_FLOAT16_VALUE(   727.50), SIMDE_FLOAT16_VALUE(    61.84), SIMDE_FLOAT16_VALUE(   534.00), SIMDE_FLOAT16_VALUE(   816.50),
        SIMDE_FLOAT16_VALUE(   322.25), SIMDE_FLOAT16_VALUE(  -819.00), SIMDE_FLOAT16_VALUE(   840.50), SIMDE_FLOAT16_VALUE(  -541.50),
        SIMDE_FLOAT16_VALUE(  -336.75), SIMDE_FLOAT16_VALUE(  -387.50), SIMDE_FLOAT16_VALUE(  -515.50), SIMDE_FLOAT16_VALUE(   982.50) },
      SIMDE_FLOAT16_VALUE(   982.50) },
    { { SIMDE_FLOAT16_VALUE(   380.50), SIMDE_FLOAT16_VALUE(   673.00), SIMDE_FLOAT16_VALUE(  -754.50), SIMDE_FLOAT16_VALUE(  -980.00),
        SIMDE_FLOAT16_VALUE(   733.50), SIMDE_FLOAT16_VALUE(  -140.88), SIMDE_FLOAT16_VALUE(   400.25), SIMDE_FLOAT16_VALUE(    34.25),
        SIMDE_FLOAT16_VALUE(   756.50), SIMDE_FLOAT16_VALUE(  -388.75), SIMDE_FLOAT16_VALUE(   638.50), SIMDE_FLOAT16_VALUE(   183.38),
        SIMDE_FLOAT16_VALUE(  -286.75), SIMDE_FLOAT16_VALUE(  -220.38), SIMDE_FLOAT16_VALUE(   817.00), SIMDE_FLOAT16_VALUE(  -210.75),
        SIMDE_FLOAT16_VALUE(   233.00), SIMDE_FLOAT16_VALUE(   518.50), SIMDE_FLOAT16_VALUE(   723.50), SIMDE_FLOAT16_VALUE(   -39.44),
        SIMDE_FLOAT16_VALUE(  -419.75), SIMDE_FLOAT16_VALUE(   257.50), SIMDE_FLOAT16_VALUE(  -222.75), SIMDE_FLOAT16_VALUE(   902.50),
        SIMDE_FLOAT16_VALUE(   438.75), SIMDE_FLOAT16_VALUE(  -382.50), SIMDE_FLOAT16_VALUE(  -639.00), SIMDE_FLOAT16_VALUE(  -898.00),
        SIMDE_FLOAT16_VALUE(   230.12), SIMDE_FLOAT16_VALUE(  -154.12), SIMDE_FLOAT16_VALUE(  -915.50), SIMDE_FLOAT16_VALUE(  -389.50) },
      SIMDE_FLOAT16_VALUE(   902.50) },
    { { SIMDE_FLOAT16_VALUE(  -481.00), SIMDE_FLOAT16_VALUE(  -670.00), SIMDE_FLOAT16_VALUE(  -369.25), SIMDE_FLOAT16_VALUE(  -747.50),
        SIMDE_FLOAT16_VALUE(   189.25), SIMDE_FLOAT16_VALUE(  -969.00), SIMDE_FLOAT16_VALUE(   286.75), SIMDE_FLOAT16_VALUE(   -54.47),
        SIMDE_FLOAT16_VALUE(  -357.75), SIMDE_FLOAT16_VALUE(   -74.56), SIMDE_FLOAT16_VALUE(  -871.00), SIMDE_FLOAT16_VALUE(   355.50),
        SIMDE_FLOAT16_VALUE(   705.00), SIMDE_FLOAT16_VALUE(   946.00), SIMDE_FLOAT16_VALUE(  -855.00), SIMDE_FLOAT16_VALUE(   -61.97),
        SIMDE_FLOAT16_VALUE(   464.50), SIMDE_FLOAT16_VALUE(   868.00), SIMDE_FLOAT16_VALUE(   898.50), SIMDE_FLOAT16_VALUE(  -955.50),
        SIMDE_FLOAT16_VALUE(   125.81), SIMDE_FLOAT16_VALUE(  -324.25), SIMDE_FLOAT16_VALUE(   947.00), SIMDE_FLOAT16_VALUE(  -435.25),
        SIMDE_FLOAT16_VALUE(   293.50), SIMDE_FLOAT16_VALUE(  -691.50), SIMDE_FLOAT16_VALUE(  -333.50), SIMDE_FLOAT16_VALUE(  -476.50),
        SIMDE_FLOAT16_VALUE(   154.25), SIMDE_FLOAT16_VALUE(  -248.88), SIMDE_FLOAT16_VALUE(   134.12), SIMDE_FLOAT16_VALUE(   673.50) },
      SIMDE_FLOAT16_VALUE(   947.00) },
    { { SIMDE_FLOAT16_VALUE(    81.25), SIMDE_FLOAT16_VALUE(   765.00), SIMDE_FLOAT16_VALUE(   926.00), SIMDE_FLOAT16_VALUE(  -729.50),
        SIMDE_FLOAT16_VALUE(   796.00), SIMDE_FLOAT16_VALUE(   212.62), SIMDE_FLOAT16_VALUE(   216.00), SIMDE_FLOAT16_VALUE(  -562.00),
        SIMDE_FLOAT16_VALUE(  -862.00), SIMDE_FLOAT16_VALUE(   345.00), SIMDE_FLOAT16_VALUE(   794.00), SIMDE_FLOAT16_VALUE(   843.00),
        SIMDE_FLOAT16_VALUE(   291.00), SIMDE_FLOAT16_VALUE(   938.50), SIMDE_FLOAT16_VALUE(  -218.88), SIMDE_FLOAT16_VALUE(  -244.38),
        SIMDE_FLOAT16_VALUE(   807.00), SIMDE_FLOAT16_VALUE(  -320.25), SIMDE_FLOAT16_VALUE(  -199.62), SIMDE_FLOAT16_VALUE(   -67.38),
        SIMDE_FLOAT16_VALUE(   355.50), SIMDE_FLOAT16_VALUE(  -252.38), SIMDE_FLOAT16_VALUE(   497.25), SIMDE_FLOAT16_VALUE(  -351.00),
        SIMDE_FLOAT16_VALUE(    56.03), SIMDE_FLOAT16_VALUE(  -836.00), SIMDE_FLOAT16_VALUE(   172.50), SIMDE_FLOAT16_VALUE(  -789.50),
        SIMDE_FLOAT16_VALUE(   -85.00), SIMDE_FLOAT16_VALUE(  -693.50), SIMDE_FLOAT16_VALUE(   883.50), SIMDE_FLOAT16_VALUE(   996.00) },
      SIMDE_FLOAT16_VALUE(   996.00) },
    { { SIMDE_FLOAT16_VALUE(  -928.50), SIMDE_FLOAT16_VALUE(   809.50), SIMDE_FLOAT16_VALUE(  -733.50), SIMDE_FLOAT16_VALUE(   867.50),
        SIMDE_FLOAT16_VALUE(    22.09), SIMDE_FLOAT16_VALUE(   482.75), SIMDE_FLOAT16_VALUE(  -694.50), SIMDE_FLOAT16_VALUE(   160.12),
        SIMDE_FLOAT16_VALUE(  -172.25), SIMDE_FLOAT16_VALUE(  -901.00), SIMDE_FLOAT16_VALUE(     3.27), SIMDE_FLOAT16_VALUE(  -881.00),
        SIMDE_FLOAT16_VALUE(  -962.00), SIMDE_FLOAT16_VALUE(   784.50), SIMDE_FLOAT16_VALUE(  -125.56), SIMDE_FLOAT16_VALUE(   844.50),
        SIMDE_FLOAT16_VALUE(  -536.00), SIMDE_FLOAT16_VALUE(   675.00), SIMDE_FLOAT16_VALUE(  -222.75), SIMDE_FLOAT16_VALUE(   820.00),
        SIMDE_FLOAT16_VALUE(  -577.50), SIMDE_FLOAT16_VALUE(  -725.50), SIMDE_FLOAT16_VALUE(  -531.00), SIMDE_FLOAT16_VALUE(   478.50),
        SIMDE_FLOAT16_VALUE(  -561.50), SIMDE_FLOAT16_VALUE(   641.50), SIMDE_FLOAT16_VALUE(   689.00), SIMDE_FLOAT16_VALUE(   353.50),
        SIMDE_FLOAT16_VALUE(   948.00), SIMDE_FLOAT16_VALUE(   572.50), SIMDE_FLOAT16_VALUE(   349.75), SIMDE_FLOAT16_VALUE(  -980.50) },
      SIMDE_FLOAT16_VALUE(   948.00) },
    { { SIMDE_FLOAT16_VALUE(   381.75), SIMDE_FLOAT16_VALUE(   616.50), SIMDE_FLOAT16_VALUE(   887.00), SIMDE_FLOAT16_VALUE(  -596.00),
        SIMDE_FLOAT16_VALUE(    99.12), SIMDE_FLOAT16_VALUE(  -807.50), SIMDE_FLOAT16_VALUE(   564.00), SIMDE_FLOAT16_VALUE(   927.00),
        SIMDE_FLOAT16_VALUE(  -708.50), SIMDE_FLOAT16_VALUE(  -432.75), SIMDE_FLOAT16_VALUE(  -954.50), SIMDE_FLOAT16_VALUE(  -670.50),
        SIMDE_FLOAT16_VALUE(  -648.00), SIMDE_FLOAT16_VALUE(   -79.81), SIMDE_FLOAT16_VALUE(  -826.00), SIMDE_FLOAT16_VALUE(  -184.00),
        SIMDE_FLOAT16_VALUE(  -405.00), SIMDE_FLOAT16_VALUE(   -48.72), SIMDE_FLOAT16_VALUE(  -364.25), SIMDE_FLOAT16_VALUE(    17.39),
        SIMDE_FLOAT16_VALUE(   225.88), SIMDE_FLOAT16_VALUE(   104.50), SIMDE_FLOAT16_VALUE(  -504.25), SIMDE_FLOAT16_VALUE(   664.50),
        SIMDE_FLOAT16_VALUE(  -254.25), SIMDE_FLOAT16_VALUE(  -815.50), SIMDE_FLOAT16_VALUE(    17.67), SIMDE_FLOAT16_VALUE(  -306.25),
        SIMDE_FLOAT16_VALUE(   757.00), SIMDE_FLOAT16_VALUE(  -632.50), SIMDE_FLOAT16_VALUE(  -286.75), SIMDE_FLOAT16_VALUE(   138.88) },
      SIMDE_FLOAT16_VALUE(   927.00) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512h a = simde_mm512_loadu_ph(test_vec[i].a);
    simde_float16 r = simde_mm512_reduce_max_ph(a);
    simde_assert_equal_f16(r, test_vec[i].r, 1);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512h a = simde_test_x86_random_f16x32(SIMDE_FLOAT16_VALUE(-1000.0), SIMDE_FLOAT16_VALUE(1000.0));
    simde_float16 r = simde_mm512_reduce_max_ph(a);

    simde_test_x86_write_f16x32(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_f16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_min_ph (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float16 a[32];
    const simde_float16 r;
  } test_vec[] = {
    { { SIMDE_FLOAT16_VALUE(   844.50), SIMDE_FLOAT16_VALUE(  -114.81), SIMDE_FLOAT16_VALUE(  -922.00), SIMDE_FLOAT16_VALUE(  -790.50),
        SIMDE_FLOAT16_VALUE(  -598.00), SIMDE_FLOAT16_VALUE(   970.00), SIMDE_FLOAT16_VALUE(   305.00), SIMDE_FLOAT16_VALUE(  -214.62),
        SIMDE_FLOAT16_VALUE(   798.00), SIMDE_FLOAT16_VALUE(   346.25), SIMDE_FLOAT16_VALUE(   477.00), SIMDE_FLOAT16_VALUE(   573.00),
        SIMDE_FLOAT16_VALUE(   247.00), SIMDE_FLOAT16_VALUE(  -550.00), SIMDE_FLOAT16_VALUE(  -505.25), SIMDE_FLOAT16_VALUE(   688.00),
        SIMDE_FLOAT16_VALUE(   446.75), SIMDE_FLOAT16_VALUE(  -639.00), SIMDE_FLOAT16_VALUE(  -624.00), SIMDE_FLOAT16_VALUE(   541.00),
        SIMDE_FLOAT16_VALUE(   638.50), SIMDE_FLOAT16_VALUE(   643.50), SIMDE_FLOAT16_VALUE(   948.50), SIMDE_FLOAT16_VALUE(   328.50),
        SIMDE_FLOAT16_VALUE(  -432.25), SIMDE_FLOAT16_VALUE(  -517.50), SIMDE_FLOAT16_VALUE(  -562.50), SIMDE_FLOAT16_VALUE(   839.50),
        SIMDE_FLOAT16_VALUE(  -883.00), SIMDE_FLOAT16_VALUE(   450.25), SIMDE_FLOAT16_VALUE(  -607.50), SIMDE_FLOAT16_VALUE(   962.00) },
      SIMDE_FLOAT16_VALUE(  -922.00) },
    { { SIMDE_FLOAT16_VALUE(  -664.50), SIMDE_FLOAT16_VALUE(  -530.00), SIMDE_FLOAT16_VALUE(  -829.00), SIMDE_FLOAT16_VALUE(  -262.50),
        SIMDE_FLOAT16_VALUE(  -560.00), SIMDE_FLOAT16_VALUE(   476.25), SIMDE_FLOAT16_VALUE(   523.00), SIMDE_FLOAT16_VALUE(  -762.00),
        SIMDE_FLOAT16_VALUE(  -177.62), SIMDE_FLOAT16_VALUE(     0.05), SIMDE_FLOAT16_VALUE(   811.00), SIMDE_FLOAT16_VALUE(  -930.50),
        SIMDE_FLOAT16_VALUE(   450.00), SIMDE_FLOAT16_VALUE(  -694.00), SIMDE_FLOAT16_VALUE(   757.50), SIMDE_FLOAT16_VALUE(  -103.44),
        SIMDE_FLOAT16_VALUE(  -333.00), SIMDE_FLOAT16_VALUE(  -866.50), SIMDE_FLOAT16_VALUE(  -562.50), SIMDE_FLOAT16_VALUE(  -694.50),
        SIMDE_FLOAT16_VALUE(   776.50), SIMDE_FLOAT16_VALUE(  -614.00), SIMDE_FLOAT16_VALUE(   634.00), SIMDE_FLOAT16_VALUE(  -655.50),
        SIMDE_FLOAT16_VALUE(  -131.38), SIMDE_FLOAT16_VALUE(  -928.50), SIMDE_FLOAT16_VALUE(  -816.00), SIMDE_FLOAT16_VALUE(   -14.12),
        SIMDE_FLOAT16_VALUE(   521.50), SIMDE_FLOAT16_VALUE(  -423.50), SIMDE_FLOAT16_VALUE(   -52.28), SIMDE_FLOAT16_VALUE(   857.00) },
      SIMDE_FLOAT16_VALUE(  -930.50) },
    { { SIMDE_FLOAT16_VALUE(    46.66), SIMDE_FLOAT16_VALUE(   118.94), SIMDE_FLOAT16_VALUE(  -405.50), SIMDE_FLOAT16_VALUE(   486.75),
        SIMDE_FLOAT16_VALUE(  -404.75), SIMDE_FLOAT16_VALUE(  -882.50), SIMDE_FLOAT16_VALUE(   724.50), SIMDE_FLOAT16_VALUE(   417.50),
        SIMDE_FLOAT16_VALUE(   117.50), SIMDE_FLOAT16_VALUE(   536.00), SIMDE_FLOAT16_VALUE(   487.00), SIMDE_FLOAT16_VALUE(  -432.75),
        SIMDE_FLOAT16_VALUE(   842.00), SIMDE_FLOAT16_VALUE(   244.38), SIMDE_FLOAT16_VALUE(   464.00), SIMDE_FLOAT16_VALUE(  -491.25),
        SIMDE_FLOAT16_VALUE(   377.75), SIMDE_FLOAT16_VALUE(   901.50), SIMDE_FLOAT16_VALUE(  -185.75), SIMDE_FLOAT16_VALUE(   154.38),
        SIMDE_FLOAT16_VALUE(  -712.50), SIMDE_FLOAT16_VALUE(  -551.50), SIMDE_FLOAT16_VALUE(   498.75), SIMDE_FLOAT16_VALUE(   156.38),
        SIMDE_FLOAT16_VALUE(  -480.25), SIMDE_FLOAT16_VALUE(   683.00), SIMDE_FLOAT16_VALUE(  -858.00), SIMDE_FLOAT16_VALUE(  -959.00),
        SIMDE_FLOAT16_VALUE(  -741.00), SIMDE_FLOAT16_VALUE(    90.00), SIMDE_FLOAT16_VALUE(   898.00), SIMDE_FLOAT16_VALUE(   306.00) },
      SIMDE_FLOAT16_VALUE(  -959.00) },
    { { SIMDE_FLOAT16_VALUE(  -791.00), SIMDE_FLOAT16_VALUE(  -507.50), SIMDE_FLOAT16_VALUE(  -207.38), SIMDE_FLOAT16_VALUE(  -195.88),
        SIMDE_FLOAT16_VALUE(  -390.00), SIMDE_FLOAT16_VALUE(  -482.75), SIMDE_FLOAT16_VALUE(  -778.50), SIMDE_FLOAT16_VALUE(   727.50),
        SIMDE_FLOAT16_VALUE(  -946.50), SIMDE_FLOAT16_VALUE(   708.50), SIMDE_FLOAT16_VALUE(  -705.00), SIMDE_FLOAT16_VALUE(   895.00),
        SIMDE_FLOAT16_VALUE(   -46.94), SIMDE_FLOAT16_VALUE(   758.50), SIMDE_FLOAT16_VALUE(  -596.00), SIMDE_FLOAT16_VALUE(  -669.00),
        SIMDE_FLOAT16_VALUE(   660.00), SIMDE_FLOAT16_VALUE(   218.25), SIMDE_FLOAT16_VALUE(   485.25), SIMDE_FLOAT16_VALUE(   948.00),
        SIMDE_FLOAT16_VALUE(   666.50), SIMDE_FLOAT16_VALUE(   -16.06), SIMDE_FLOAT16_VALUE(   104.25), SIMDE_FLOAT16_VALUE(  -813.50),
        SIMDE_FLOAT16_VALUE(  -333.25), SIMDE_FLOAT16_VALUE(   246.50), SIMDE_FLOAT16_VALUE(  -772.50), SIMDE_FLOAT16_VALUE(   -74.06),
        SIMDE_FLOAT16_VALUE(  -663.50), SIMDE_FLOAT16_VALUE(  -874.50), SIMDE_FLOAT16_VALUE(  -768.00), SIMDE_FLOAT16_VALUE(  -454.50) },
      SIMDE_FLOAT16_VALUE(  -946.50) },
    { { SIMDE_FLOAT16_VALUE(  -382.00), SIMDE_FLOAT16_VALUE(    24.48), SIMDE_FLOAT16_VALUE(   349.50), SIMDE_FLOAT16_VALUE(   228.00),
        SIMDE_FLOAT16_VALUE(   542.00), SIMDE_FLOAT16_VALUE(   571.00), SIMDE_FLOAT16_VALUE(   -44.53), SIMDE_FLOAT16_VALUE(   595.00),
        SIMDE_FLOAT16_VALUE(   279.75), SIMDE_FLOAT16_VALUE(   250.25), SIMDE_FLOAT16_VALUE(   490.25), SIMDE_FLOAT16_VALUE(  -767.00),
        SIMDE_FLOAT16_VALUE(     8.94), SIMDE_FLOAT16_VALUE(   894.00), SIMDE_FLOAT16_VALUE(  -436.25), SIMDE_FLOAT16_VALUE(  -330.75),
        SIMDE_FLOAT16_VALUE(   112.38), SIMDE_FLOAT16_VALUE(  -951.00), SIMDE_FLOAT16_VALUE(  -383.00), SIMDE_FLOAT16_VALUE(  -221.12),
        SIMDE_FLOAT16_VALUE(    32.81), SIMDE_FLOAT16_VALUE(   721.50), SIMDE_FLOAT16_VALUE(   -34.78), SIMDE_FLOAT16_VALUE(   699.50),
        SIMDE_FLOAT16_VALUE(   -32.19), SIMDE_FLOAT16_VALUE(   192.75), SIMDE_FLOAT16_VALUE(  -374.50), SIMDE_FLOAT16_VALUE(   304.25),
        SIMDE_FLOAT16_VALUE(   318.25), SIMDE_FLOAT16_VALUE(  -142.62), SIMDE_FLOAT16_VALUE(   849.50), SIMDE_FLOAT16_VALUE(   936.50) },
      SIMDE_FLOAT16_VALUE(  -951.00) },
    { { SIMDE_FLOAT16_VALUE(   882.00), SIMDE_FLOAT16_VALUE(   199.12), SIMDE_FLOAT16_VALUE(   164.50), SIMDE_FLOAT16_VALUE(   423.75),
        SIMDE_FLOAT16_VALUE(  -229.62), SIMDE_FLOAT16_VALUE(  -880.00), SIMDE_FLOAT16_VALUE(    18.75), SIMDE_FLOAT16_VALUE(  -950.00),
        SIMDE_FLOAT16_VALUE(   370.25), SIMDE_FLOAT16_VALUE(  -491.00), SIMDE_FLOAT16_VALUE(  -717.00), SIMDE_FLOAT16_VALUE(  -621.00),
        SIMDE_FLOAT16_VALUE(  -597.00), SIMDE_FLOAT16_VALUE(  -153.25), SIMDE_FLOAT16_VALUE(    48.25), SIMDE_FLOAT16_VALUE(   515.50),
        SIMDE_FLOAT16_VALUE(  -104.31), SIMDE_FLOAT16_VALUE(   665.50), SIMDE_FLOAT16_VALUE(  -705.50), SIMDE_FLOAT16_VALUE(   928.50),
        SIMDE_FLOAT16_VALUE(   386.50), SIMDE_FLOAT16_VALUE(   259.75), SIMDE_FLOAT16_VALUE(   628.00), SIMDE_FLOAT16_VALUE(  -645.50),
        SIMDE_FLOAT16_VALUE(  -547.50), SIMDE_FLOAT16_VALUE(  -746.50), SIMDE_FLOAT16_VALUE(   658.50), SIMDE_FLOAT16_VALUE(   770.50),
        SIMDE_FLOAT16_VALUE(   110.81), SIMDE_FLOAT16_VALUE(   508.50), SIMDE_FLOAT16_VALUE(   707.00), SIMDE_FLOAT16_VALUE(    -7.41) },
      SIMDE_FLOAT16_VALUE(  -950.00) },
    { { SIMDE_FLOAT16_VALUE(  -292.50), SIMDE_FLOAT16_VALUE(  -128.38), SIMDE_FLOAT16_VALUE(  -584.00), SIMDE_FLOAT16_VALUE(   478.00),
        SIMDE_FLOAT16_VALUE(    -8.43), SIMDE_FLOAT16_VALUE(   435.00), SIMDE_FLOAT16_VALUE(   528.00), SIMDE_FLOAT16_VALUE(  -638.00),
        SIMDE_FLOAT16_VALUE(   944.00), SIMDE_FLOAT16_VALUE(   811.00), SIMDE_FLOAT16_VALUE(  -259.00), SIMDE_FLOAT16_VALUE(  -653.00),
        SIMDE_FLOAT16_VALUE(  -342.00), SIMDE_FLOAT16_VALUE(   789.00), SIMDE_FLOAT16_VALUE(   862.50), SIMDE_FLOAT16_VALUE(   553.50),
        SIMDE_FLOAT16_VALUE(   454.50), SIMDE_FLOAT16_VALUE(  -843.00), SIMDE_FLOAT16_VALUE(   482.00), SIMDE_FLOAT16_VALUE(  -159.00),
        SIMDE_FLOAT16_VALUE(   416.75), SIMDE_FLOAT16_VALUE(   110.12), SIMDE_FLOAT16_VALUE(   195.38), SIMDE_FLOAT16_VALUE(   869.00),
        SIMDE_FLOAT16_VALUE(   363.50), SIMDE_FLOAT16_VALUE(  -146.00), SIMDE_FLOAT16_VALUE(   640.00), SIMDE_FLOAT16_VALUE(  -525.50),
        SIMDE_FLOAT16_VALUE(  -637.50), SIMDE_FLOAT16_VALUE(   347.00), SIMDE_FLOAT16_VALUE(   467.00), SIMDE_FLOAT16_VALUE(    70.00) },
      SIMDE_FLOAT16_VALUE(  -843.00) },
    { { SIMDE_FLOAT16_VALUE(  -781.50), SIMDE_FLOAT16_VALUE(   883.00), SIMDE_FLOAT16_VALUE(  -452.00), SIMDE_FLOAT16_VALUE(   210.25),
        SIMDE_FLOAT16_VALUE(   318.25), SIMDE_FLOAT16_VALUE(  -924.00), SIMDE_FLOAT16_VALUE(   572.00), SIMDE_FLOAT16_VALUE(   262.25),
        SIMDE_FLOAT16_VALUE(   887.00), SIMDE_FLOAT16_VALUE(  -687.00), SIMDE_FLOAT16_VALUE(   609.50), SIMDE_FLOAT16_VALUE(  -454.75),
        SIMDE_FLOAT16_VALUE(  -898.00), SIMDE_FLOAT16_VALUE(   472.00), SIMDE_FLOAT16_VALUE(  -901.00), SIMDE_FLOAT16_VALUE(   556.50),
        SIMDE_FLOAT16_VALUE(   629.00), SIMDE_FLOAT16_VALUE(   581.00), SIMDE_FLOAT16_VALUE(  -602.50), SIMDE_FLOAT16_VALUE(    46.00),
        SIMDE_FLOAT16_VALUE(  -309.00), SIMDE_FLOAT16_VALUE(   593.00), SIMDE_FLOAT16_VALUE(   -84.81), SIMDE_FLOAT16_VALUE(  -945.50),
        SIMDE_FLOAT16_VALUE(  -553.00), SIMDE_FLOAT16_VALUE(  -445.00), SIMDE_FLOAT16_VALUE(  -471.00), SIMDE_FLOAT16_VALUE(  -190.75),
        SIMDE_FLOAT16_VALUE(   902.00), SIMDE_FLOAT16_VALUE(   996.00), SIMDE_FLOAT16_VALUE(   879.50), SIMDE_FLOAT16_VALUE(  -879.00) },
      SIMDE_FLOAT16_VALUE(  -945.50) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512h a = simde_mm512_loadu_ph(test_vec[i].a);
    simde_float16 r = simde_mm512_reduce_min_ph(a);
    simde_assert_equal_f16(r, test_vec[i].r, 1);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512h a = simde_test_x86_random_f16x32(SIMDE_FLOAT16_VALUE(-1000.0), SIMDE_FLOAT16_VALUE(1000.0));
    simde_float16 r = simde_mm512_reduce_min_ph(a);

    simde_test_x86_write_f16x32(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_f16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}
#endif // SIMDE_FLOAT16_IS_SCALAR

static int
test_simde_mm512_reduce_max_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[16];
    int32_t r;
  } test_vec[] = {
    { { -INT32_C(   813481257),  INT32_C(   546817848), -INT32_C(  1757264748),  INT32_C(   769874228),  INT32_C(   460183175), -INT32_C(   678992654),  INT32_C(     4370223),  INT32_C(  1518532483),
         INT32_C(  1915356473),  INT32_C(  1637007821),  INT32_C(   989385734), -INT32_C(  1251484626), -INT32_C(  1513040718),  INT32_C(  1736202040), -INT32_C(  1989689850), -INT32_C(   270210379) },
       INT32_C(  1915356473) },
    { { -INT32_C(  1134489873), -INT32_C(   719457329), -INT32_C(   166783288), -INT32_C(  1532266766), -INT32_C(  2109113526), -INT32_C(   655702574),  INT32_C(   962744707),  INT32_C(   724059707),
         INT32_C(   602376532),  INT32_C(  1157170556),  INT32_C(   221906971), -INT32_C(   911088257),  INT32_C(   843905888),  INT32_C(  1141585600), -INT32_C(  1015190137),  INT32_C(   133080499) },
       INT32_C(  1157170556) },
    { { -INT32_C(  1440033490), -INT32_C(   168877094), -INT32_C(  1425921748),  INT32_C(  1869919247),  INT32_C(  1872937135),  INT32_C(  2125704694), -INT32_C(   851365862),  INT32_C(    64237525),
        -INT32_C(   542179836),  INT32_C(  1322556706), -INT32_C(   705046586),  INT32_C(   977628555),  INT32_C(   615114542), -INT32_C(  1365090924),  INT32_C(  1669063565),  INT32_C(   392580882) },
       INT32_C(  2125704694) },
    { {  INT32_C(  1878398029),  INT32_C(  2025704113),  INT32_C(   743290529),  INT32_C(  1382453796),  INT32_C(   225841273), -INT32_C(    88401555),  INT32_C(   257767164), -INT32_C(   752434043),
        -INT32_C(  1975378728), -INT32_C(  2013134618), -INT32_C(   625717322),  INT32_C(  1529617378), -INT32_C(  1737907413), -INT32_C(  1198381892), -INT32_C(   523767717), -INT32_C(  1934365260) },
       INT32_C(  2025704113) },
    { { -INT32_C(   266930679), -INT32_C(  1401415434),  INT32_C(  1250372712),  INT32_C(  1923461959),  INT32_C(   302648918), -INT32_C(  1899258573),  INT32_C(  1097765517), -INT32_C(  1983044992),
         INT32_C(   242869272),  INT32_C(  1689973244),  INT32_C(  1705918750),  INT32_C(  1272468725), -INT32_C(  1788943774),  INT32_C(   203630975),  INT32_C(   994939579), -INT32_C(   859497548) },
       INT32_C(  1923461959) },
    { { -INT32_C(    69583105),  INT32_C(  1314952240), -INT32_C(   877457706), -INT32_C(   988378270), -INT32_C(   312839058),  INT32_C(  1509588637), -INT32_C(   996915440),  INT32_C(  1636850018),
        -INT32_C(   933401960), -INT32_C(   736707074),  INT32_C(   782223819), -INT32_C(  1007438251), -INT32_C(   927970006), -INT32_C(   618550581),  INT32_C(  1419752946), -INT32_C(  1481232625) },
       INT32_C(  1636850018) },
    { { -INT32_C(  1737550951), -INT32_C(  1687386672), -INT32_C(  1547105202), -INT32_C(   328811326), -INT32_C(   709617910), -INT32_C(  1280191039), -INT32_C(  1710731125),  INT32_C(   423739008),
        -INT32_C(  1582190383), -INT32_C(  2093212107), -INT32_C(   332986838), -INT32_C(   858223166),  INT32_C(  1721929124), -INT32_C(   317107358),  INT32_C(   596058531), -INT32_C(  1338128161) },
       INT32_C(  1721929124) },
    { { -INT32_C(  1387139464),  INT32_C(   909217292),  INT32_C(  1445091476), -INT32_C(  1977353243), -INT32_C(   353319544), -INT32_C(  1143469800),  INT32_C(   182411051), -INT32_C(  1598350296),
         INT32_C(   390990858),  INT32_C(   776830874), -INT32_C(  1132105513), -INT32_C(   213473173), -INT32_C(  2049100179),  INT32_C(  1799402816),  INT32_C(  1031085845),  INT32_C(  1188900923) },
       INT32_C(  1799402816) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    int32_t r = simde_mm512_reduce_max_epi32(a);
    simde_assert_equal_i32(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_i32x16();
    int32_t r = simde_mm512_reduce_max_epi32(a);

    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_max_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[8];
    int64_t r;
  } test_vec[] = {
    { { -INT64_C( 7414994043216169726), -INT64_C( 7605818785096979682),  INT64_C( 8143719191631782972), -INT64_C( 1937506738173131693),
         INT64_C( 6330432945267905037),  INT64_C( 1735163336545740898),  INT64_C( 8346079544720478957), -INT64_C( 4987558819094325760) },
       INT64_C( 8346079544720478957) },
    { { -INT64_C( 9096191391897426125),  INT64_C( 5832746886490020606),  INT64_C( 7312303613956729575), -INT64_C( 3241729026166292581),
        -INT64_C( 3127745723160789000), -INT64_C( 7740947043993020194),  INT64_C( 7320062202497156299), -INT64_C( 1762962511517262060) },
       INT64_C( 7320062202497156299) },
    { { -INT64_C( 6242948083026573135),  INT64_C( 8678574663337184047),  INT64_C( 3886957366707030977),  INT64_C(  285259843876069150),
        -INT64_C( 4994519976962755526), -INT64_C( 2876103514003105039),  INT64_C( 2119319649943600915), -INT64_C(   51473889080159244) },
       INT64_C( 8678574663337184047) },
    { { -INT64_C( 6099364031695039724),  INT64_C(  809909028452430158), -INT64_C( 1496697051629510370), -INT64_C( 7295415206456236548),
        -INT64_C( 3984679092976236979),  INT64_C( 8829425114441186436),  INT64_C( 7921499077913069580), -INT64_C( 1849024954399514826) },
       INT64_C( 8829425114441186436) },
    { { -INT64_C( 6342579952647499849), -INT64_C( 6956744673328879459),  INT64_C( 1392615992519166837),  INT64_C( 3111747048904295343),
         INT64_C( 6593085956190707125), -INT64_C( 4129016558291409792), -INT64_C( 7654430629652834860),  INT64_C( 8007977084563330178) },
       INT64_C( 8007977084563330178) },
    { {  INT64_C( 6288088135339835177),  INT64_C( 3855395061599213674), -INT64_C( 5770047711820173966),  INT64_C(   29775698208410176),
        -INT64_C( 2200524676799156153),  INT64_C( 8095535882043253186),  INT64_C( 3181992064485638220), -INT64_C( 2426155323914465666) },
       INT64_C( 8095535882043253186) },
    { {  INT64_C( 2997553912019237953), -INT64_C( 6790683502249757100), -INT64_C( 4828348912931009819), -INT64_C(  683027937610423023),
         INT64_C( 5345852979216810657),  INT64_C(  609216452785700222), -INT64_C( 2097595708107813143), -INT64_C(  385040890218285828) },
       INT64_C( 5345852979216810657) },
    { {  INT64_C( 2148835632955348881), -INT64_C( 5126954761824669777), -INT64_C( 8671747556685925388), -INT64_C( 3383644509044133705),
        -INT64_C( 1372144066370204872),  INT64_C( 4548801973779883673),  INT64_C( 8745053429732540234), -INT64_C( 5770074887091022372) },
       INT64_C( 8745053429732540234) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    int64_t r = simde_mm512_reduce_max_epi64(a);
    simde_assert_equal_i64(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_i64x8();
    int64_t r = simde_mm512_reduce_max_epi64(a);

    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_max_epu32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const uint32_t a[16];
    uint32_t r;
  } test_vec[] = {
    { { UINT32_C( 343028045), UINT32_C(3568367038), UINT32_C(1711941916), UINT32_C(1858791686), UINT32_C(1934254942), UINT32_C( 861200181), UINT32_C(2206274930), UINT32_C(3805222293),
        UINT32_C(  83247430), UINT32_C(1389930038), UINT32_C(3585663695), UINT32_C( 692355787), UINT32_C( 463245030), UINT32_C(3746492525), UINT32_C(2607009541), UINT32_C( 712848100) },
      UINT32_C(3805222293) },
    { { UINT32_C(2855171187), UINT32_C(3942385178), UINT32_C(3015685608), UINT32_C( 501023543), UINT32_C(4265114001), UINT32_C(1876854634), UINT32_C( 990527830), UINT32_C(3865413747),
        UINT32_C( 378574076), UINT32_C(2197851546), UINT32_C(2033631298), UINT32_C(1435964355), UINT32_C(4149464973), UINT32_C(2909155670), UINT32_C(3857215858), UINT32_C(4123807225) },
      UINT32_C(4265114001) },
    { { UINT32_C(2081119457), UINT32_C( 738135273), UINT32_C(2409969100), UINT32_C(3605281865), UINT32_C(1657616395), UINT32_C(3691983721), UINT32_C(2646734756), UINT32_C( 647138628),
        UINT32_C(3550650090), UINT32_C(1996464554), UINT32_C( 520463574), UINT32_C(3958762208), UINT32_C(2337128994), UINT32_C(2590465270), UINT32_C(2553817427), UINT32_C(2696858294) },
      UINT32_C(3958762208) },
    { { UINT32_C( 309616744), UINT32_C(3616043777), UINT32_C(4160196119), UINT32_C(2598562936), UINT32_C(2770677935), UINT32_C(3762261388), UINT32_C(1819834294), UINT32_C(2852992578),
        UINT32_C(2545779094), UINT32_C( 191776244), UINT32_C(1258448339), UINT32_C(  31843410), UINT32_C(2712013588), UINT32_C(1317135768), UINT32_C(2679830877), UINT32_C(3309946671) },
      UINT32_C(4160196119) },
    { { UINT32_C(1012663880), UINT32_C( 524798540), UINT32_C(2188069168), UINT32_C(1115901998), UINT32_C(4091750747), UINT32_C(1799447566), UINT32_C(2349530205), UINT32_C( 206656707),
        UINT32_C(2789780826), UINT32_C(2831585400), UINT32_C( 120205785), UINT32_C(3712658817), UINT32_C(3838848470), UINT32_C(4015002258), UINT32_C(3514587662), UINT32_C( 165531054) },
      UINT32_C(4091750747) },
    { { UINT32_C(4088342139), UINT32_C(2409330102), UINT32_C( 681035174), UINT32_C(1208344946), UINT32_C(2687292686), UINT32_C(4119887079), UINT32_C(2244414679), UINT32_C(1418634457),
        UINT32_C(2152152778), UINT32_C(1511056051), UINT32_C( 427992999), UINT32_C(2522974088), UINT32_C(1144491612), UINT32_C(3778660106), UINT32_C(2892431571), UINT32_C(1862399397) },
      UINT32_C(4119887079) },
    { { UINT32_C(3891284019), UINT32_C(3527475243), UINT32_C( 804045735), UINT32_C(2797948234), UINT32_C(3874159835), UINT32_C(2529633475), UINT32_C(3376623140), UINT32_C(1463370788),
        UINT32_C(3074304396), UINT32_C(3498737449), UINT32_C(2365552194), UINT32_C(2670970051), UINT32_C(2223316417), UINT32_C(1713065025), UINT32_C(2670681723), UINT32_C( 787900578) },
      UINT32_C(3891284019) },
    { { UINT32_C(3135649169), UINT32_C(4153045172), UINT32_C(2843970022), UINT32_C( 256423502), UINT32_C( 362008020), UINT32_C(2507910682), UINT32_C(2922687244), UINT32_C(2782734867),
        UINT32_C( 341820255), UINT32_C( 420211251), UINT32_C(3250753139), UINT32_C( 433064773), UINT32_C(4063126744), UINT32_C( 528984594), UINT32_C(1758313301), UINT32_C(1158523622) },
      UINT32_C(4153045172) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    uint32_t r = simde_mm512_reduce_max_epu32(a);
    simde_assert_equal_u32(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_u32x16();
    uint32_t r = simde_mm512_reduce_max_epu32(a);

    simde_test_x86_write_u32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_u32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_max_epu64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const uint64_t a[8];
    uint64_t r;
  } test_vec[] = {
    { { UINT64_C(12696813536712629165), UINT64_C( 7789281467542717539), UINT64_C( 4525567160522043380), UINT64_C(11227443008793002145),
        UINT64_C( 8562164309004119934), UINT64_C( 5139803803857927599), UINT64_C( 1713891746710645295), UINT64_C( 3981527010801014199) },
      UINT64_C(12696813536712629165) },
    { { UINT64_C(10570560629856925736), UINT64_C( 5189218905104243049), UINT64_C(15908573315603770681), UINT64_C(10475306114705666160),
        UINT64_C( 5634706707999209916), UINT64_C(13094512215385091160), UINT64_C(10897351171457190970), UINT64_C( 4347697128206744923) },
      UINT64_C(15908573315603770681) },
    { { UINT64_C(18017510826358612952), UINT64_C( 6767543693926405562), UINT64_C(15451840953525751729), UINT64_C( 2698202965668998836),
        UINT64_C( 1563572998833851669), UINT64_C(17546646755910294786), UINT64_C(16849719055174286982), UINT64_C( 8708992338335986025) },
      UINT64_C(18017510826358612952) },
    { { UINT64_C(12559661590589193947), UINT64_C( 6067850417307621375), UINT64_C(11867314028861980819), UINT64_C( 4140838159734460316),
        UINT64_C( 9750217913778808821), UINT64_C(  897903492646642515), UINT64_C(10510705488852994558), UINT64_C( 7199857700817090645) },
      UINT64_C(12559661590589193947) },
    { { UINT64_C( 7619901548923214767), UINT64_C( 9582533587170396279), UINT64_C(12343974425522000073), UINT64_C(10855163734446170067),
        UINT64_C( 8441026627282774692), UINT64_C( 8583685717822320148), UINT64_C( 5490313974412270896), UINT64_C( 8925909986216676962) },
      UINT64_C(12343974425522000073) },
    { { UINT64_C( 5571051943594732765), UINT64_C( 4984648152413418760), UINT64_C( 5163655034814816996), UINT64_C( 5178417649256773258),
        UINT64_C( 8548521956102837019), UINT64_C( 8467429829253040680), UINT64_C(15341646587303644604), UINT64_C(  171721619091650736) },
      UINT64_C(15341646587303644604) },
    { { UINT64_C( 2380990199054697767), UINT64_C( 5587244825958037379), UINT64_C( 9605288313324722734), UINT64_C(13562310543210938958),
        UINT64_C(  765712574167748230), UINT64_C( 1040516090731363986), UINT64_C( 9393332572190563842), UINT64_C( 9486533534634474324) },
      UINT64_C(13562310543210938958) },
    { { UINT64_C(16505589632491255677), UINT64_C(15532486122730434208), UINT64_C( 4219320784552596695), UINT64_C(12863627665166517764),
        UINT64_C(14481543576753243840), UINT64_C( 5882499881872392437), UINT64_C( 6776487360312725738), UINT64_C(13265884829110447665) },
      UINT64_C(16505589632491255677) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    uint64_t r = simde_mm512_reduce_max_epu64(a);
    simde_assert_equal_u64(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_u64x8();
    uint64_t r = simde_mm512_reduce_max_epu64(a);

    simde_test_x86_write_u64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_u64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_max_ps (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float32 a[16];
    simde_float32 r;
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -227.80), SIMDE_FLOAT32_C(  -813.20), SIMDE_FLOAT32_C(  -774.12), SIMDE_FLOAT32_C(   352.82),
        SIMDE_FLOAT32_C(   578.16), SIMDE_FLOAT32_C(  -784.99), SIMDE_FLOAT32_C(   476.83), SIMDE_FLOAT32_C(  -137.24),
        SIMDE_FLOAT32_C(   387.10), SIMDE_FLOAT32_C(  -103.78), SIMDE_FLOAT32_C(   749.70), SIMDE_FLOAT32_C(  -117.98),
        SIMDE_FLOAT32_C(   705.36), SIMDE_FLOAT32_C(   729.23), SIMDE_FLOAT32_C(  -858.49), SIMDE_FLOAT32_C(     4.86) },
      SIMDE_FLOAT32_C(   749.70) },
    { { SIMDE_FLOAT32_C(  -623.44), SIMDE_FLOAT32_C(   632.93), SIMDE_FLOAT32_C(   553.05), SIMDE_FLOAT32_C(  -971.84),
        SIMDE_FLOAT32_C(  -455.67), SIMDE_FLOAT32_C(   249.48), SIMDE_FLOAT32_C(   -32.31), SIMDE_FLOAT32_C(   544.55),
        SIMDE_FLOAT32_C(   698.98), SIMDE_FLOAT32_C(   403.02), SIMDE_FLOAT32_C(   222.74), SIMDE_FLOAT32_C(  -155.96),
        SIMDE_FLOAT32_C(   -18.78), SIMDE_FLOAT32_C(  -866.45), SIMDE_FLOAT32_C(  -946.22), SIMDE_FLOAT32_C(   753.43) },
      SIMDE_FLOAT32_C(   753.43) },
    { { SIMDE_FLOAT32_C(  -679.65), SIMDE_FLOAT32_C(  -720.34), SIMDE_FLOAT32_C(   106.25), SIMDE_FLOAT32_C(   898.51),
        SIMDE_FLOAT32_C(  -505.33), SIMDE_FLOAT32_C(  -416.92), SIMDE_FLOAT32_C(  -238.73), SIMDE_FLOAT32_C(   881.77),
        SIMDE_FLOAT32_C(   479.30), SIMDE_FLOAT32_C(  -489.03), SIMDE_FLOAT32_C(  -236.21), SIMDE_FLOAT32_C(   184.67),
        SIMDE_FLOAT32_C(  -759.81), SIMDE_FLOAT32_C(   -94.70), SIMDE_FLOAT32_C(  -810.47), SIMDE_FLOAT32_C(  -383.24) },
      SIMDE_FLOAT32_C(   898.51) },
    { { SIMDE_FLOAT32_C(  -461.76), SIMDE_FLOAT32_C(   742.58), SIMDE_FLOAT32_C(  -355.08), SIMDE_FLOAT32_C(    82.57),
        SIMDE_FLOAT32_C(    -7.94), SIMDE_FLOAT32_C(   612.61), SIMDE_FLOAT32_C(  -372.89), SIMDE_FLOAT32_C(  -308.96),
        SIMDE_FLOAT32_C(    15.63), SIMDE_FLOAT32_C(   849.86), SIMDE_FLOAT32_C(   535.09), SIMDE_FLOAT32_C(   996.85),
        SIMDE_FLOAT32_C(   983.41), SIMDE_FLOAT32_C(   588.87), SIMDE_FLOAT32_C(   750.28), SIMDE_FLOAT32_C(  -696.25) },
      SIMDE_FLOAT32_C(   996.85) },
    { { SIMDE_FLOAT32_C(   868.54), SIMDE_FLOAT32_C(  -143.47), SIMDE_FLOAT32_C(  -797.74), SIMDE_FLOAT32_C(  -636.79),
        SIMDE_FLOAT32_C(   439.61), SIMDE_FLOAT32_C(   -36.47), SIMDE_FLOAT32_C(  -755.02), SIMDE_FLOAT32_C(   -81.08),
        SIMDE_FLOAT32_C(   474.49), SIMDE_FLOAT32_C(     8.77), SIMDE_FLOAT32_C(  -896.42), SIMDE_FLOAT32_C(   714.69),
        SIMDE_FLOAT32_C(   914.07), SIMDE_FLOAT32_C(  -706.89), SIMDE_FLOAT32_C(  -668.55), SIMDE_FLOAT32_C(  -547.69) },
      SIMDE_FLOAT32_C(   914.07) },
    { { SIMDE_FLOAT32_C(  -964.31), SIMDE_FLOAT32_C(   -23.63), SIMDE_FLOAT32_C(   534.87), SIMDE_FLOAT32_C(    27.75),
        SIMDE_FLOAT32_C(  -411.02), SIMDE_FLOAT32_C(  -838.01), SIMDE_FLOAT32_C(   718.79), SIMDE_FLOAT32_C(   604.61),
        SIMDE_FLOAT32_C(  -988.15), SIMDE_FLOAT32_C(   253.88), SIMDE_FLOAT32_C(   601.47), SIMDE_FLOAT32_C(   995.25),
        SIMDE_FLOAT32_C(  -157.24), SIMDE_FLOAT32_C(   351.75), SIMDE_FLOAT32_C(  -701.00), SIMDE_FLOAT32_C(  -288.71) },
      SIMDE_FLOAT32_C(   995.25) },
    { { SIMDE_FLOAT32_C(  -791.72), SIMDE_FLOAT32_C(  -498.74), SIMDE_FLOAT32_C(    74.50), SIMDE_FLOAT32_C(   647.89),
        SIMDE_FLOAT32_C(   464.79), SIMDE_FLOAT32_C(   319.48), SIMDE_FLOAT32_C(  -433.19), SIMDE_FLOAT32_C(   -60.72),
        SIMDE_FLOAT32_C(  -671.76), SIMDE_FLOAT32_C(  -329.60), SIMDE_FLOAT32_C(  -346.03), SIMDE_FLOAT32_C(  -757.68),
        SIMDE_FLOAT32_C(   -36.49), SIMDE_FLOAT32_C(   -14.58), SIMDE_FLOAT32_C(  -305.38), SIMDE_FLOAT32_C(    -0.80) },
      SIMDE_FLOAT32_C(   647.89) },
    { { SIMDE_FLOAT32_C(   961.78), SIMDE_FLOAT32_C(  -770.51), SIMDE_FLOAT32_C(  -973.05), SIMDE_FLOAT32_C(  -449.24),
        SIMDE_FLOAT32_C(  -608.52), SIMDE_FLOAT32_C(   745.74), SIMDE_FLOAT32_C(  -844.63), SIMDE_FLOAT32_C(  -596.67),
        SIMDE_FLOAT32_C(    -0.38), SIMDE_FLOAT32_C(   756.84), SIMDE_FLOAT32_C(  -601.42), SIMDE_FLOAT32_C(   842.38),
        SIMDE_FLOAT32_C(   108.59), SIMDE_FLOAT32_C(  -302.42), SIMDE_FLOAT32_C(  -446.33), SIMDE_FLOAT32_C(   316.87) },
      SIMDE_FLOAT32_C(   961.78) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde_float32 r = simde_mm512_reduce_max_ps(a);
    simde_assert_equal_f32(r, test_vec[i].r, 1);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde_float32 r = simde_mm512_reduce_max_ps(a);

    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_f32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_max_pd (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float64 a[8];
    simde_float64 r;
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   459.52), SIMDE_FLOAT64_C(   440.89), SIMDE_FLOAT64_C(  -235.09), SIMDE_FLOAT64_C(   407.51),
        SIMDE_FLOAT64_C(   179.69), SIMDE_FLOAT64_C(  -281.61), SIMDE_FLOAT64_C(    76.27), SIMDE_FLOAT64_C(  -950.23) },
      SIMDE_FLOAT64_C(   459.52) },
    { { SIMDE_FLOAT64_C(   718.04), SIMDE_FLOAT64_C(  -735.14), SIMDE_FLOAT64_C(   120.42), SIMDE_FLOAT64_C(  -576.22),
        SIMDE_FLOAT64_C(  -212.79), SIMDE_FLOAT64_C(  -981.04), SIMDE_FLOAT64_C(   830.57), SIMDE_FLOAT64_C(  -876.03) },
      SIMDE_FLOAT64_C(   830.57) },
    { { SIMDE_FLOAT64_C(   465.45), SIMDE_FLOAT64_C(  -568.15), SIMDE_FLOAT64_C(  -493.75), SIMDE_FLOAT64_C(   185.65),
        SIMDE_FLOAT64_C(  -605.41), SIMDE_FLOAT64_C(  -901.91), SIMDE_FLOAT64_C(  -404.42), SIMDE_FLOAT64_C(   -47.69) },
      SIMDE_FLOAT64_C(   465.45) },
    { { SIMDE_FLOAT64_C(  -830.44), SIMDE_FLOAT64_C(    -2.10), SIMDE_FLOAT64_C(   825.03), SIMDE_FLOAT64_C(   384.89),
        SIMDE_FLOAT64_C(   282.54), SIMDE_FLOAT64_C(  -224.42), SIMDE_FLOAT64_C(  -197.52), SIMDE_FLOAT64_C(  -257.94) },
      SIMDE_FLOAT64_C(   825.03) },
    { { SIMDE_FLOAT64_C(  -783.53), SIMDE_FLOAT64_C(   567.39), SIMDE_FLOAT64_C(  -850.43), SIMDE_FLOAT64_C(   396.16),
        SIMDE_FLOAT64_C(  -714.22), SIMDE_FLOAT64_C(   225.84), SIMDE_FLOAT64_C(   445.93), SIMDE_FLOAT64_C(  -996.19) },
      SIMDE_FLOAT64_C(   567.39) },
    { { SIMDE_FLOAT64_C(   490.70), SIMDE_FLOAT64_C(  -433.65), SIMDE_FLOAT64_C(  -572.41), SIMDE_FLOAT64_C(  -722.09),
        SIMDE_FLOAT64_C(  -414.69), SIMDE_FLOAT64_C(  -741.84), SIMDE_FLOAT64_C(  -598.12), SIMDE_FLOAT64_C(  -949.24) },
      SIMDE_FLOAT64_C(   490.70) },
    { { SIMDE_FLOAT64_C(  -309.98), SIMDE_FLOAT64_C(   -91.87), SIMDE_FLOAT64_C(   236.42), SIMDE_FLOAT64_C(    84.60),
        SIMDE_FLOAT64_C(     6.22), SIMDE_FLOAT64_C(   832.00), SIMDE_FLOAT64_C(  -963.09), SIMDE_FLOAT64_C(   175.77) },
      SIMDE_FLOAT64_C(   832.00) },
    { { SIMDE_FLOAT64_C(  -170.11), SIMDE_FLOAT64_C(   861.94), SIMDE_FLOAT64_C(  -439.33), SIMDE_FLOAT64_C(  -887.56),
        SIMDE_FLOAT64_C(  -362.48), SIMDE_FLOAT64_C(   363.15), SIMDE_FLOAT64_C(  -145.50), SIMDE_FLOAT64_C(  -146.00) },
      SIMDE_FLOAT64_C(   861.94) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde_float64 r = simde_mm512_reduce_max_pd(a);
    simde_assert_equal_f64(r, test_vec[i].r, 1);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde_float64 r = simde_mm512_reduce_max_pd(a);

    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_f64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_min_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[16];
    int32_t r;
  } test_vec[] = {
    { {  INT32_C(   566551290), -INT32_C(  1323765948), -INT32_C(  1751293148),  INT32_C(  1255507603), -INT32_C(  1340035191),  INT32_C(   449616932), -INT32_C(  1739109821),  INT32_C(   725375793),
        -INT32_C(  2125725635),  INT32_C(   338846960),  INT32_C(  1537986503), -INT32_C(   559579051),  INT32_C(  1418642735), -INT32_C(  1536271519), -INT32_C(   700594779),  INT32_C(  1560377632) },
      -INT32_C(  2125725635) },
    { {  INT32_C(  1792953978),  INT32_C(  2055082162),  INT32_C(   886385119), -INT32_C(   653034838), -INT32_C(  1590845120), -INT32_C(  1572430852), -INT32_C(  2139585696),  INT32_C(  1994226428),
         INT32_C(  2061548488), -INT32_C(  1426825525),  INT32_C(   853526920), -INT32_C(  2079591868), -INT32_C(  1876608621),  INT32_C(   892496852), -INT32_C(   340415762), -INT32_C(   329149660) },
      -INT32_C(  2139585696) },
    { {  INT32_C(   442974798),  INT32_C(   683957152),  INT32_C(  1767547685),  INT32_C(   703424149),  INT32_C(  1941508767),  INT32_C(  1823009662), -INT32_C(  1168679275),  INT32_C(  1067891185),
        -INT32_C(  1688662533), -INT32_C(  1899749783),  INT32_C(  1475813057),  INT32_C(   595649668),  INT32_C(  1956067830), -INT32_C(  1176420572), -INT32_C(  1905051491), -INT32_C(   322037007) },
      -INT32_C(  1905051491) },
    { { -INT32_C(  1870125273),  INT32_C(   119426117), -INT32_C(   279046806), -INT32_C(   267198727),  INT32_C(   996452631), -INT32_C(  2030811671),  INT32_C(  1863673982), -INT32_C(  1453530238),
         INT32_C(  1346036746), -INT32_C(  1705551824),  INT32_C(  1737078126), -INT32_C(  1437098861),  INT32_C(   786807877),  INT32_C(  2142624001), -INT32_C(  1007695295), -INT32_C(  1217573971) },
      -INT32_C(  2030811671) },
    { {  INT32_C(  1594337071),  INT32_C(  1845059327), -INT32_C(  1495956717),  INT32_C(  1682975775), -INT32_C(   359451160),  INT32_C(  1349077007), -INT32_C(  1089185774), -INT32_C(   763985501),
         INT32_C(   657554984), -INT32_C(   258659364), -INT32_C(   845780306),  INT32_C(  2133976982),  INT32_C(   728351772),  INT32_C(   511496716), -INT32_C(   824340437),  INT32_C(   966874129) },
      -INT32_C(  1495956717) },
    { { -INT32_C(  1369386286), -INT32_C(  1415645699), -INT32_C(   159894177),  INT32_C(   947235356),  INT32_C(  2070208110), -INT32_C(   610672464), -INT32_C(  2119534736), -INT32_C(  1648735285),
         INT32_C(   457906717),  INT32_C(  1875307024),  INT32_C(   996491039),  INT32_C(  1467210473),  INT32_C(  1775425464),  INT32_C(   658795703), -INT32_C(  1364660509),  INT32_C(  1464558137) },
      -INT32_C(  2119534736) },
    { { -INT32_C(  1921870211), -INT32_C(  1610860416),  INT32_C(  1624924791), -INT32_C(   172470724), -INT32_C(   581006811), -INT32_C(   654007562), -INT32_C(   897077871), -INT32_C(  1943940593),
        -INT32_C(   384199831),  INT32_C(  1133057740), -INT32_C(  1264295048), -INT32_C(   676766543), -INT32_C(   575404058),  INT32_C(  1001830570),  INT32_C(  1963277925),  INT32_C(  2030118672) },
      -INT32_C(  1943940593) },
    { { -INT32_C(  2040325190), -INT32_C(  1446319055),  INT32_C(    22900303), -INT32_C(  1328019510), -INT32_C(  1198683122), -INT32_C(  1426897852), -INT32_C(  1843398270), -INT32_C(   636739552),
         INT32_C(  1818324795), -INT32_C(  1441453221),  INT32_C(  1672180633), -INT32_C(  2011987078),  INT32_C(  1413587215),  INT32_C(  1744712933),  INT32_C(  1308171565),  INT32_C(  2032665917) },
      -INT32_C(  2040325190) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    int32_t r = simde_mm512_reduce_min_epi32(a);
    simde_assert_equal_i32(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_i32x16();
    int32_t r = simde_mm512_reduce_min_epi32(a);

    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_min_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[8];
    int64_t r;
  } test_vec[] = {
    { {  INT64_C( 5583051442079762804), -INT64_C( 5156406438818470546),  INT64_C( 1418645866806030692),  INT64_C( 2656712216608352807),
         INT64_C( 3176002828375409426), -INT64_C( 1444382768898980973),  INT64_C( 3688699930697007414),  INT64_C( 3198248540477886863) },
      -INT64_C( 5156406438818470546) },
    { {  INT64_C( 6207092525676779065), -INT64_C(  848497803350885944), -INT64_C( 2105923948543644024),  INT64_C( 1015706666489896658),
        -INT64_C( 2885916689683838247), -INT64_C( 7775923200908229277),  INT64_C( 4180165535136753613),  INT64_C( 9092555051630993997) },
      -INT64_C( 7775923200908229277) },
    { { -INT64_C( 4382549580443863736),  INT64_C( 9166227886227936147), -INT64_C( 7438643338688898014),  INT64_C( 8136009086626556612),
        -INT64_C( 6610451847536245986),  INT64_C( 4934389532889196360),  INT64_C( 9203267170938526421),  INT64_C( 8724828569662485116) },
      -INT64_C( 7438643338688898014) },
    { {  INT64_C( 1768008997605530580),  INT64_C( 2073782591220338273), -INT64_C( 6637191876296469080),  INT64_C( 8159497358015974503),
        -INT64_C( 3522132194911439818),  INT64_C( 7498803171072029357), -INT64_C( 6090926889645683026),  INT64_C( 5803074049394196346) },
      -INT64_C( 6637191876296469080) },
    { {  INT64_C( 7717609272109358813),  INT64_C( 3733045958898904230), -INT64_C( 4346677018582541633), -INT64_C( 6725734873528919233),
         INT64_C( 1903624566097182529), -INT64_C(  347534472918217868), -INT64_C(  651425772057315607),  INT64_C( 8371339802839076806) },
      -INT64_C( 6725734873528919233) },
    { { -INT64_C( 7599861606734685448),  INT64_C( 4747003291971356841),  INT64_C( 5737700865749540596), -INT64_C( 2171290370756186057),
        -INT64_C( 2749213169262112480),  INT64_C( 1978662602036701661), -INT64_C( 8622873417633896323),  INT64_C( 6134131619603296086) },
      -INT64_C( 8622873417633896323) },
    { {  INT64_C( 2318068768134622753),  INT64_C(  511944201703123672), -INT64_C( 7166525229679910733),  INT64_C( 8006550549042003346),
        -INT64_C(  302045590810741910), -INT64_C( 6888794470754933803),  INT64_C( 7554559497107436533),  INT64_C( 1686962746025816749) },
      -INT64_C( 7166525229679910733) },
    { { -INT64_C( 1505166456478422274),  INT64_C( 2325177009099642550), -INT64_C( 4295816015992068061), -INT64_C( 2546459182186392860),
        -INT64_C( 5236814447854653870),  INT64_C( 8105247108977368367), -INT64_C( 1268945228792146041), -INT64_C( 7126901011494460833) },
      -INT64_C( 7126901011494460833) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    int64_t r = simde_mm512_reduce_min_epi64(a);
    simde_assert_equal_i64(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_i64x8();
    int64_t r = simde_mm512_reduce_min_epi64(a);

    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_min_epu32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const uint32_t a[16];
    uint32_t r;
  } test_vec[] = {
    { { UINT32_C(2567247256), UINT32_C(1582389294), UINT32_C(1119464693), UINT32_C(3534894155), UINT32_C( 359669003), UINT32_C(1224987626), UINT32_C(1082491893), UINT32_C(1155374764),
        UINT32_C(3806257843), UINT32_C( 792735546), UINT32_C( 645003739), UINT32_C( 972563245), UINT32_C( 709781568), UINT32_C( 829641020), UINT32_C(2154953172), UINT32_C(1254444695) },
      UINT32_C( 359669003) },
    { { UINT32_C(1781310256), UINT32_C(2912513234), UINT32_C(2480081509), UINT32_C(1858915118), UINT32_C(1872304691), UINT32_C(1067519083), UINT32_C(2629767685), UINT32_C(2447869025),
        UINT32_C(4194046759), UINT32_C(3852899712), UINT32_C(3464001952), UINT32_C(2017215556), UINT32_C(3387413854), UINT32_C(3876096226), UINT32_C(4236560539), UINT32_C(1955425100) },
      UINT32_C(1067519083) },
    { { UINT32_C(4268591486), UINT32_C(3219395615), UINT32_C(3532479629), UINT32_C(4283091361), UINT32_C(2177446303), UINT32_C(1432932794), UINT32_C(3864128665), UINT32_C(3596279383),
        UINT32_C(2278868840), UINT32_C(1766242523), UINT32_C(3057373973), UINT32_C(1001751964), UINT32_C(1891401398), UINT32_C(3922011472), UINT32_C(1775179281), UINT32_C(1564420597) },
      UINT32_C(1001751964) },
    { { UINT32_C(3437499377), UINT32_C(3778357964), UINT32_C(2576838909), UINT32_C(2899660021), UINT32_C( 454857163), UINT32_C(3355763382), UINT32_C(3979465976), UINT32_C(3997855997),
        UINT32_C(1337667203), UINT32_C(1429270616), UINT32_C(1458554721), UINT32_C(3741500436), UINT32_C( 217718613), UINT32_C(4191485441), UINT32_C(3504801234), UINT32_C(4173214069) },
      UINT32_C( 217718613) },
    { { UINT32_C(3091757408), UINT32_C(3406723178), UINT32_C(1411513664), UINT32_C( 372450497), UINT32_C(1143090499), UINT32_C(4265473579), UINT32_C(1892558075), UINT32_C(3060370518),
        UINT32_C(1886302470), UINT32_C(1765506089), UINT32_C( 985488505), UINT32_C(3276927104), UINT32_C(1225224989), UINT32_C(1699169642), UINT32_C(3218478441), UINT32_C(2809479073) },
      UINT32_C( 372450497) },
    { { UINT32_C( 420996336), UINT32_C(3666039392), UINT32_C( 789856431), UINT32_C(1324574001), UINT32_C(1134033625), UINT32_C(2846416447), UINT32_C(2506653427), UINT32_C(2906447549),
        UINT32_C( 583488706), UINT32_C(1442597542), UINT32_C(3146060170), UINT32_C(1326086262), UINT32_C(2995954034), UINT32_C(1935358848), UINT32_C(1997063097), UINT32_C(1663321505) },
      UINT32_C( 420996336) },
    { { UINT32_C(1065806745), UINT32_C(3231023669), UINT32_C( 175839891), UINT32_C(  72975762), UINT32_C(2813783079), UINT32_C(3759804711), UINT32_C(1985422293), UINT32_C(  31095912),
        UINT32_C(2638241895), UINT32_C(1985861090), UINT32_C(2172705007), UINT32_C(2240207198), UINT32_C(3962322117), UINT32_C( 600655438), UINT32_C(3516474473), UINT32_C( 148009888) },
      UINT32_C(  31095912) },
    { { UINT32_C(3064271827), UINT32_C(3626762984), UINT32_C( 945401050), UINT32_C(1270734725), UINT32_C(1782049052), UINT32_C(2576155696), UINT32_C(3379242537), UINT32_C(1842429338),
        UINT32_C( 958625360), UINT32_C(1393643384), UINT32_C(2173397755), UINT32_C(1724664138), UINT32_C(1657799474), UINT32_C( 838622472), UINT32_C( 502949763), UINT32_C(4119579812) },
      UINT32_C( 502949763) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    uint32_t r = simde_mm512_reduce_min_epu32(a);
    simde_assert_equal_u32(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_u32x16();
    uint32_t r = simde_mm512_reduce_min_epu32(a);

    simde_test_x86_write_u32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_u32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_min_epu64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const uint64_t a[8];
    uint64_t r;
  } test_vec[] = {
    { { UINT64_C(17946351928412646978), UINT64_C( 1538338688285186473), UINT64_C(  757576183074728266), UINT64_C(10120910291343998426),
        UINT64_C( 9968812201639125601), UINT64_C( 7198677560287220718), UINT64_C(12362724924684852481), UINT64_C(14739048392179759542) },
      UINT64_C(  757576183074728266) },
    { { UINT64_C( 1633826068423758400), UINT64_C(  204013289211374167), UINT64_C( 8693948129919797546), UINT64_C(12685518031048659412),
        UINT64_C( 1374581661455497429), UINT64_C(17414390374188972712), UINT64_C( 2745951476175943393), UINT64_C( 7748006196793479781) },
      UINT64_C(  204013289211374167) },
    { { UINT64_C( 9113451481732241578), UINT64_C( 3831951216702127477), UINT64_C( 1477730082944981513), UINT64_C(  326070451542026118),
        UINT64_C(16004636843054919367), UINT64_C( 2336040091939128289), UINT64_C(13629577160753365858), UINT64_C( 4990093382917820131) },
      UINT64_C(  326070451542026118) },
    { { UINT64_C( 8112670196996746061), UINT64_C( 6518979119443980353), UINT64_C(15152520304798724948), UINT64_C(14799478732859777183),
        UINT64_C( 9653775058400986618), UINT64_C(  224148294821846729), UINT64_C(13189308383230854984), UINT64_C( 7008505263121766646) },
      UINT64_C(  224148294821846729) },
    { { UINT64_C(12269463314289575804), UINT64_C(13921619823410488193), UINT64_C(10294668646449463567), UINT64_C( 7269451812451955226),
        UINT64_C( 3501666014127770638), UINT64_C(11152764852041221261), UINT64_C(15815410171708030806), UINT64_C(15320752003464593233) },
      UINT64_C( 3501666014127770638) },
    { { UINT64_C(13468396412576449290), UINT64_C( 3856636952203709023), UINT64_C( 4098731846666438469), UINT64_C( 9635469527058890013),
        UINT64_C( 3943508228091837395), UINT64_C( 5009875255299242610), UINT64_C( 7209185786234382574), UINT64_C( 7091199706281836137) },
      UINT64_C( 3856636952203709023) },
    { { UINT64_C( 4650280862236256124), UINT64_C( 4673636156801395607), UINT64_C(13576700667129767872), UINT64_C( 3427531456591737503),
        UINT64_C(12231507282743194800), UINT64_C( 2902602292492521910), UINT64_C( 2113482894141775172), UINT64_C(12623270892840744268) },
      UINT64_C( 2113482894141775172) },
    { { UINT64_C( 8253952133295943889), UINT64_C( 3301072460190501950), UINT64_C(16481796504093589024), UINT64_C(13095361487669757711),
        UINT64_C( 1652297526001855169), UINT64_C( 5478221220497156591), UINT64_C(17584205673660074891), UINT64_C(13567022390155864749) },
      UINT64_C( 1652297526001855169) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    uint64_t r = simde_mm512_reduce_min_epu64(a);
    simde_assert_equal_u64(r, test_vec[i].r);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512i a = simde_test_x86_random_u64x8();
    uint64_t r = simde_mm512_reduce_min_epu64(a);

    simde_test_x86_write_u64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_u64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_min_ps (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float32 a[16];
    simde_float32 r;
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(    23.88), SIMDE_FLOAT32_C(  -394.98), SIMDE_FLOAT32_C(  -337.60), SIMDE_FLOAT32_C(   448.65),
        SIMDE_FLOAT32_C(   394.23), SIMDE_FLOAT32_C(   -42.09), SIMDE_FLOAT32_C(  -242.50), SIMDE_FLOAT32_C(  -896.14),
        SIMDE_FLOAT32_C(   614.19), SIMDE_FLOAT32_C(  -304.36), SIMDE_FLOAT32_C(    58.80), SIMDE_FLOAT32_C(   464.53),
        SIMDE_FLOAT32_C(   655.43), SIMDE_FLOAT32_C(   825.60), SIMDE_FLOAT32_C(   454.22), SIMDE_FLOAT32_C(  -577.87) },
      SIMDE_FLOAT32_C(  -896.14) },
    { { SIMDE_FLOAT32_C(   130.11), SIMDE_FLOAT32_C(   524.75), SIMDE_FLOAT32_C(   955.85), SIMDE_FLOAT32_C(   -59.90),
        SIMDE_FLOAT32_C(   585.38), SIMDE_FLOAT32_C(   145.67), SIMDE_FLOAT32_C(   -74.43), SIMDE_FLOAT32_C(   -10.85),
        SIMDE_FLOAT32_C(    83.80), SIMDE_FLOAT32_C(   735.99), SIMDE_FLOAT32_C(   936.68), SIMDE_FLOAT32_C(  -462.75),
        SIMDE_FLOAT32_C(  -180.18), SIMDE_FLOAT32_C(  -559.41), SIMDE_FLOAT32_C(   198.33), SIMDE_FLOAT32_C(   843.71) },
      SIMDE_FLOAT32_C(  -559.41) },
    { { SIMDE_FLOAT32_C(    45.61), SIMDE_FLOAT32_C(   860.74), SIMDE_FLOAT32_C(   292.36), SIMDE_FLOAT32_C(  -560.16),
        SIMDE_FLOAT32_C(  -181.35), SIMDE_FLOAT32_C(  -950.14), SIMDE_FLOAT32_C(  -456.30), SIMDE_FLOAT32_C(  -567.16),
        SIMDE_FLOAT32_C(  -254.50), SIMDE_FLOAT32_C(   602.50), SIMDE_FLOAT32_C(   897.36), SIMDE_FLOAT32_C(  -599.07),
        SIMDE_FLOAT32_C(   428.10), SIMDE_FLOAT32_C(   351.58), SIMDE_FLOAT32_C(  -176.95), SIMDE_FLOAT32_C(  -441.79) },
      SIMDE_FLOAT32_C(  -950.14) },
    { { SIMDE_FLOAT32_C(  -123.67), SIMDE_FLOAT32_C(  -221.09), SIMDE_FLOAT32_C(   498.31), SIMDE_FLOAT32_C(  -538.29),
        SIMDE_FLOAT32_C(   924.58), SIMDE_FLOAT32_C(  -576.11), SIMDE_FLOAT32_C(   450.86), SIMDE_FLOAT32_C(     8.38),
        SIMDE_FLOAT32_C(  -840.12), SIMDE_FLOAT32_C(   387.54), SIMDE_FLOAT32_C(   545.64), SIMDE_FLOAT32_C(   -20.30),
        SIMDE_FLOAT32_C(   828.13), SIMDE_FLOAT32_C(  -256.03), SIMDE_FLOAT32_C(  -176.59), SIMDE_FLOAT32_C(  -126.25) },
      SIMDE_FLOAT32_C(  -840.12) },
    { { SIMDE_FLOAT32_C(  -395.29), SIMDE_FLOAT32_C(  -884.23), SIMDE_FLOAT32_C(   313.59), SIMDE_FLOAT32_C(   423.35),
        SIMDE_FLOAT32_C(  -834.37), SIMDE_FLOAT32_C(   857.28), SIMDE_FLOAT32_C(   856.19), SIMDE_FLOAT32_C(   -88.88),
        SIMDE_FLOAT32_C(   459.78), SIMDE_FLOAT32_C(   753.55), SIMDE_FLOAT32_C(   312.05), SIMDE_FLOAT32_C(  -112.12),
        SIMDE_FLOAT32_C(   105.13), SIMDE_FLOAT32_C(  -864.89), SIMDE_FLOAT32_C(   446.09), SIMDE_FLOAT32_C(   981.46) },
      SIMDE_FLOAT32_C(  -884.23) },
    { { SIMDE_FLOAT32_C(   -85.99), SIMDE_FLOAT32_C(   -55.60), SIMDE_FLOAT32_C(  -556.83), SIMDE_FLOAT32_C(  -161.41),
        SIMDE_FLOAT32_C(   368.28), SIMDE_FLOAT32_C(   894.03), SIMDE_FLOAT32_C(   846.97), SIMDE_FLOAT32_C(   528.17),
        SIMDE_FLOAT32_C(   281.57), SIMDE_FLOAT32_C(   392.61), SIMDE_FLOAT32_C(  -492.13), SIMDE_FLOAT32_C(   109.70),
        SIMDE_FLOAT32_C(  -863.42), SIMDE_FLOAT32_C(   331.28), SIMDE_FLOAT32_C(   983.45), SIMDE_FLOAT32_C(  -258.71) },
      SIMDE_FLOAT32_C(  -863.42) },
    { { SIMDE_FLOAT32_C(   447.05), SIMDE_FLOAT32_C(   297.03), SIMDE_FLOAT32_C(  -835.36), SIMDE_FLOAT32_C(   612.68),
        SIMDE_FLOAT32_C(   154.31), SIMDE_FLOAT32_C(  -979.17), SIMDE_FLOAT32_C(  -476.20), SIMDE_FLOAT32_C(  -385.91),
        SIMDE_FLOAT32_C(   774.38), SIMDE_FLOAT32_C(   835.86), SIMDE_FLOAT32_C(   501.97), SIMDE_FLOAT32_C(  -120.49),
        SIMDE_FLOAT32_C(   970.96), SIMDE_FLOAT32_C(   -51.94), SIMDE_FLOAT32_C(  -139.03), SIMDE_FLOAT32_C(  -115.03) },
      SIMDE_FLOAT32_C(  -979.17) },
    { { SIMDE_FLOAT32_C(   892.46), SIMDE_FLOAT32_C(   304.14), SIMDE_FLOAT32_C(   723.56), SIMDE_FLOAT32_C(   260.74),
        SIMDE_FLOAT32_C(   198.16), SIMDE_FLOAT32_C(   570.54), SIMDE_FLOAT32_C(  -211.09), SIMDE_FLOAT32_C(  -520.27),
        SIMDE_FLOAT32_C(   -36.85), SIMDE_FLOAT32_C(   296.78), SIMDE_FLOAT32_C(   589.43), SIMDE_FLOAT32_C(    99.73),
        SIMDE_FLOAT32_C(  -371.95), SIMDE_FLOAT32_C(   572.88), SIMDE_FLOAT32_C(   841.02), SIMDE_FLOAT32_C(  -924.90) },
      SIMDE_FLOAT32_C(  -924.90) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde_float32 r = simde_mm512_reduce_min_ps(a);
    simde_assert_equal_f32(r, test_vec[i].r, 1);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde_float32 r = simde_mm512_reduce_min_ps(a);

    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_f32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_reduce_min_pd (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float64 a[8];
    simde_float64 r;
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -130.09), SIMDE_FLOAT64_C(  -994.34), SIMDE_FLOAT64_C(   687.78), SIMDE_FLOAT64_C(  -975.78),
        SIMDE_FLOAT64_C(  -973.51), SIMDE_FLOAT64_C(  -788.41), SIMDE_FLOAT64_C(  -361.68), SIMDE_FLOAT64_C(   800.87) },
      SIMDE_FLOAT64_C(  -994.34) },
    { { SIMDE_FLOAT64_C(  -952.56), SIMDE_FLOAT64_C(  -859.71), SIMDE_FLOAT64_C(  -319.62), SIMDE_FLOAT64_C(  -981.60),
        SIMDE_FLOAT64_C(    88.34), SIMDE_FLOAT64_C(   541.34), SIMDE_FLOAT64_C(   -96.62), SIMDE_FLOAT64_C(   -19.20) },
      SIMDE_FLOAT64_C(  -981.60) },
    { { SIMDE_FLOAT64_C(  -154.52), SIMDE_FLOAT64_C(  -373.06), SIMDE_FLOAT64_C(  -758.46), SIMDE_FLOAT64_C(  -956.36),
        SIMDE_FLOAT64_C(  -802.52), SIMDE_FLOAT64_C(    30.45), SIMDE_FLOAT64_C(  -476.63), SIMDE_FLOAT64_C(   160.63) },
      SIMDE_FLOAT64_C(  -956.36) },
    { { SIMDE_FLOAT64_C(  -672.78), SIMDE_FLOAT64_C(  -887.20), SIMDE_FLOAT64_C(  -739.65), SIMDE_FLOAT64_C(   -44.72),
        SIMDE_FLOAT64_C(   685.68), SIMDE_FLOAT64_C(  -898.63), SIMDE_FLOAT64_C(    30.38), SIMDE_FLOAT64_C(  -444.41) },
      SIMDE_FLOAT64_C(  -898.63) },
    { { SIMDE_FLOAT64_C(  -892.97), SIMDE_FLOAT64_C(  -281.83), SIMDE_FLOAT64_C(  -420.19), SIMDE_FLOAT64_C(  -866.48),
        SIMDE_FLOAT64_C(   -70.25), SIMDE_FLOAT64_C(   218.13), SIMDE_FLOAT64_C(   934.39), SIMDE_FLOAT64_C(   -22.81) },
      SIMDE_FLOAT64_C(  -892.97) },
    { { SIMDE_FLOAT64_C(   358.41), SIMDE_FLOAT64_C(  -385.23), SIMDE_FLOAT64_C(    -4.40), SIMDE_FLOAT64_C(  -553.24),
        SIMDE_FLOAT64_C(  -843.89), SIMDE_FLOAT64_C(   898.97), SIMDE_FLOAT64_C(   427.56), SIMDE_FLOAT64_C(     1.59) },
      SIMDE_FLOAT64_C(  -843.89) },
    { { SIMDE_FLOAT64_C(  -474.09), SIMDE_FLOAT64_C(   669.10), SIMDE_FLOAT64_C(    45.24), SIMDE_FLOAT64_C(  -276.61),
        SIMDE_FLOAT64_C(  -300.46), SIMDE_FLOAT64_C(   568.61), SIMDE_FLOAT64_C(   884.01), SIMDE_FLOAT64_C(    26.76) },
      SIMDE_FLOAT64_C(  -474.09) },
    { { SIMDE_FLOAT64_C(   681.41), SIMDE_FLOAT64_C(  -855.63), SIMDE_FLOAT64_C(   982.04), SIMDE_FLOAT64_C(   367.09),
        SIMDE_FLOAT64_C(  -754.26), SIMDE_FLOAT64_C(    12.42), SIMDE_FLOAT64_C(   922.68), SIMDE_FLOAT64_C(  -647.23) },
      SIMDE_FLOAT64_C(  -855.63) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde_float64 r = simde_mm512_reduce_min_pd(a);
    simde_assert_equal_f64(r, test_vec[i].r, 1);
  }

  return 0;
#else
  fputc('\n', stdout);

  for (int i = 0; i < 8; i++) {
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde_float64 r = simde_mm512_reduce_min_pd(a);

    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_f64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  #if defined(SIMDE_FLOAT16_IS_SCALAR)
    SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_max_ph)
    SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_min_ph)
  #endif
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_max_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_max_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_max_epu32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_max_epu64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_max_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_max_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_min_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_min_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_min_epu32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_min_epu64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_min_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_reduce_min_pd)

SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
