/*	Dispatcher

PIRL CVS ID: Dispatcher.java,v 1.37 2012/04/16 06:11:36 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Messenger;

import	PIRL.PVL.*;
import	PIRL.Configuration.*;
import	PIRL.Utilities.Host;
import	PIRL.Utilities.Authentication;

import	java.nio.channels.ServerSocketChannel;
import	java.nio.channels.SocketChannel;
import	java.nio.channels.AsynchronousCloseException;
import	java.net.InetSocketAddress;
import	java.net.InetAddress;
import	java.net.MulticastSocket;
import	java.net.DatagramPacket;
import	java.io.File;
import	java.io.PrintStream;
import	java.io.FileOutputStream;
import	java.io.FileNotFoundException;
import	java.io.IOException;

import	java.security.KeyPair;

import	java.text.SimpleDateFormat;
import	java.util.Date;
import	java.util.Collections;
import	java.util.Vector;


/**	A <i>Dispatcher</i> is a Messenger manager for communication between
	client processes.
<p>
	A Dispatcher is typically used as a background daemon application
	process that accepts connections from clients on a designated
	communications ports, allocates a Messenger to each client, manages
	the Messengers, and handles a core set of Messenger management
	Messages. When a Distpatcher is started it will send a "Hello"
	broadcast on a designated multicast port to any clients listening
	on the local host system who might be interested in making a
	connection.
<p>
	A Dispatcher may be subclassed to add a additional Message protocol
	support
<p>
	A Configuration file is used to Configure a Dispatcher's operating
	variables before it begins running.
<p>
	A log file may be specified where client connection/deconnection
	and other important events are reported.
<p>
	Support is provided for authenticated client connections using a
	public key to exchange an encoded private password that is decoded
	by a private key paired with the public key.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.37
	@see		Messenger
*/
public class Dispatcher
	implements Message_Delivered_Listener, Runnable
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Messenger.Dispatcher (1.37 2012/04/16 06:11:36)";

/**	The ID of of the Dispatcher.
<p>
	It is initialized to the {@link #ID}. Subclasses are expected to set
	this to their own ID.
*/
protected String
	Dispatcher_ID					= ID;

/**	The default name of the Dispatcher: "Dispatcher".
*/
public static final String
	DEFAULT_DISPATCHER_NAME			= "Dispatcher";
private static String
	Default_Dispatcher_Name			= DEFAULT_DISPATCHER_NAME;
private String
	Dispatcher_Name					= null;

private Message
	Identity						= null;

private Exception
	Run_Exception					= null;

//	Configuration:

/**	The Dispatcher Configuration.
*/
protected Configuration
	The_Configuration				= null;

/**	The configuration parameter name for the communicaitons port number value.
*/
public static final String
	PORT_PARAMETER_NAME				= "Port";

/**	The configuration parameter name for the maximum wait time, in
	seconds, for a new Messenger client connection to respond to the
	identify message.
*/
public static final String
	IDENTIFY_TIMEOUT_PARAMETER_NAME	= "Identify_Timeout";

/**	The configuration parameter name for the authentication password.
*/
public static final String
	PASSWORD_PARAMETER_NAME			= "Password";

/**	The configuration parameter name for the log file pathname.
*/
public static final String
	LOG_PATHNAME_PARAMETER_NAME		= "Log_Pathname";

/**	The configuration parameter name for the Hello port.
*/
public static final String
	HELLO_PORT_PARAMETER_NAME		= "Hello_Port";

/**	The configuration parameter name for the Hello multicast address.
*/
public static final String
	HELLO_ADDRESS_PARAMETER_NAME	= "Hello_Address";

/**	The configuration parameter name for the Hello repeat rate (seconds).
*/
public static final String
	HELLO_RATE_PARAMETER_NAME		= "Hello_Repeat_Rate";


//	Communications port:

/**	The default communications port number.
*/
public static final int
	DEFAULT_PORT					= 4144;
private int
	Port							= 0;

private ServerSocketChannel
	Listener;

/**	The default port on which to send the Hello datagram.
*/
public static final int
	DEFAULT_HELLO_PORT				= 4170;
private int
	Hello_Port;

/**	The default private multicast address to use in the Hello datagram.
*/
public static final String
	DEFAULT_HELLO_ADDRESS			= "230.1.2.3";
private String
	Hello_Address;

/**	The content of the Hello broadcast datagram message.
*/
public static final String
	HELLO_MESSAGE					= "Hello";

private static final int
	HELLO_COUNT						= 12,
	HELLO_INTERVAL					= 5;

private Say_Hello
	Hello							= null;


//	Messenger clients:

/**	The default maximum amount of time, in seconds, to wait for a new
	Messenger client to respond to an identify message.
*/
public static final int
	DEFAULT_IDENTIFY_TIMEOUT		= 15;

/**	The minimum Messenger client identify timeout in seconds.
<p>
	@see	#DEFAULT_IDENTIFY_TIMEOUT
*/
public static final int
	MINIMUM_IDENTIFY_TIMEOUT		= 5;
private volatile int
	Identify_Timeout				= DEFAULT_IDENTIFY_TIMEOUT;

//	Authentication.

/**	Controls whether or not annonymous connections are allowed.
*/
public static final boolean
	UNAUTHENTICATED_CONNECTIONS_ALLOWED
		= false;

private String
	Password						= null,
	Public_Key						= null;
private KeyPair
	Keys							= null;

/**	List of connected Messenger clients.
*/
protected Vector<Messenger>
	Messengers						= new Vector<Messenger> ();

/**	List of Messenger clients that have requested connection notifications.
*/
protected Vector<Messenger>
	Report_to_Messengers			= new Vector<Messenger> ();

/**	Automatically send connection notifications.
*/
protected volatile boolean
	Report_Messenger_Connections	= true;


/**	Messenger management Message Actions and their parameters.
*/
public static final String
	ACTION_PARAMETER_NAME			= Message.ACTION_PARAMETER_NAME,
	IDENTIFY_ACTION					= Message.IDENTIFY_ACTION,
		KEY_PARAMETER_NAME
			= "Key",
	IDENTITY_ACTION					= Message.IDENTITY_ACTION,
		NAME_PARAMETER_NAME
			= Message.NAME_PARAMETER_NAME,
		ADDRESS_PARAMETER_NAME
			= Messenger.ADDRESS_PARAMETER_NAME,
		ROUTE_TO_PARAMETER_NAME
			= Message.ROUTE_TO_PARAMETER_NAME,
		CLASS_ID_PARAMETER_NAME
			= Message.CLASS_ID_PARAMETER_NAME,
	MESSENGERS_REPORT_ACTION		= "Messengers_Report",
	START_MESSENGER_REPORTING_ACTION= "Start_Messenger_Reporting",
	STOP_MESSENGER_REPORTING_ACTION	= "Stop_Messenger_Reporting",
	LINK_MESSENGER_ACTION			= "Link_to_Messenger",
	UNLINK_MESSENGER_ACTION			= "Unlink_from_Messenger",
	STATUS_REPORT_ACTION			= "Status_Report",
		CONFIGURATION_SOURCE_PARAMETER_NAME
			= "Configuration_Source",
		MESSENGER_STATUS_PARAMETER_NAME
			= "Messenger_Status",
			MESSAGES_SENT_PARAMETER_NAME
				= "Messages_Sent",
			MESSAGE_BYTES_SENT_PARAMETER_NAME
				= "Message_Bytes_Sent",
			MESSAGES_SENT_DROPPED_PARAMETER_NAME
				= "Messages_Sent_Dropped",
			MESSAGES_SENT_CORRUPTED_PARAMETER_NAME
				= "Messages_Sent_Corrupted",
			MESSAGES_RECEIVED_PARAMETER_NAME
				= "Messages_Received",
			MESSAGE_BYTES_RECEIVED_PARAMETER_NAME
				= "Message_Bytes_Received",
			MESSAGES_RECEIVED_DROPPED_PARAMETER_NAME
				= "Messages_Received_Dropped",
			MESSAGES_RECEIVED_CORRUPTED_PARAMETER_NAME
				= "Messages_Received_Corrupted",
			MESSAGES_UNFORWARDABLE_PARAMETER_NAME
				= "Messages_Unforwardable",
		MEMORY_STATUS_PARAMETER_NAME
			= "Memory_Status",
			MEMORY_AVAILABLE_PARAMETER_NAME
				= "Available_Memory",
			MEMORY_ALLOCATED_PARAMETER_NAME
				= "Allocated_Memory",
			MEMORY_FREE_PARAMETER_NAME
				= "Free_Memory",
	UNDELIVERABLE_ACTION			= Messenger.UNDELIVERABLE_ACTION,
	ACK_ACTION						= Message.ACK_ACTION,
	NACK_ACTION						= Message.NACK_ACTION,
		ORIGINAL_MESSAGE_PARAMETER_NAME
			= Messenger.ORIGINAL_MESSAGE_PARAMETER_NAME,
	DONE_ACTION						= Message.DONE_ACTION,
		EXPLANATION_PARAMETER_NAME
			= Messenger.EXPLANATION_PARAMETER_NAME,
		EXCEPTION_PARAMETER_NAME
			= Messenger.EXCEPTION_PARAMETER_NAME;

//	Logging:

private boolean
	Logging							= false;
private String
	Log_Pathname					= Default_Log_Filename ();

/**	Log "filename" for logging to stdout.
<p>
	@see	#Log_Pathname(String)
*/
public static final String
	STDOUT							= "-";

/**	Log output stream.
*/
protected PrintStream
	Log_Stream						= null;

/**	Formatting pattern for log message timestamps.
<p>
	@see	#DATE_FORMAT
*/
public static final String
	DATE_FORMATTING					= "d MMMMM yyyy kk:mm:ss.SSS Z";

/**	Format for log message timestamps.
<p>
	@see	#DATE_FORMATTING
*/
public static final SimpleDateFormat
	DATE_FORMAT						= new SimpleDateFormat (DATE_FORMATTING);

/**	Log file marker written when a Dispatcher is started and after it has
	shutdown.
*/
public static final String
	START_STOP_MARKER
		= "==================================================";

//	Miscellaneous:

//	The Java virtual machine (JVM) runtime environment (JRE) object.
private static final Runtime
	JRE								= Runtime.getRuntime ();

/**	Text new-line sequence.
*/
public static final String
	NL								= System.getProperty ("line.separator");


/**	Success exit status.
*/
public static final int
	EXIT_SUCCESS					= 0;

/**	Command line syntax problem exit status.
*/
public static final int
	EXIT_COMMAND_LINE_SYNTAX		= 1;

/**	The log file could not be opened for writing exit status.
*/
public static final int
	EXIT_CONFIG_FILE_PROBLEM		= 2;

/**	The log file could not be opened for writing exit status.
*/
public static final int
	EXIT_NO_LOG_FILE				= 3;

/**	IOException failure exit status.
*/
public static final int
	EXIT_IO_ERROR					= 4;

/**	SecurityException exit status.
*/
public static final int
	EXIT_SECURITY_VIOLATION			= 5;

/**	Unknown exception (possible programming flaw).
*/
public static final int
	EXIT_UNKNOWN_EXCEPTION			= 6;


// Debug control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTOR		= 1 << 1,
	DEBUG_CONFIG			= 1 << 2,
	DEBUG_BIND				= 1 << 3,
	DEBUG_CONNECT			= 1 << 4,
	DEBUG_AUTHENTICATE		= 1 << 5,
	DEBUG_MESSAGES			= 1 << 6,
	DEBUG_MESSENGERS		= 1 << 7,
	DEBUG_IDENTITIES		= 1 << 8,
	DEBUG_START_CONDUCTOR	= 1 << 9,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a named Dispatcher with a configuration and a specified
	communication port.
<p>
	<b.N.B.</b>: Constructing a Dispatcher does not {@link #Start() start}
	it operating.
<p>
	@param	name	The {@link #Dispatcher_Name(String) Dispatcher name}.
	@param	configuration	The Configuration to use to {@link
		#Configure(Configuration) configure} the Dispatcher when it is
		{@link #Start() started}.
	@param	port	The communications port number. If less than or equal
		to zero the value of the configuration {@link #PORT_PARAMETER_NAME}
		parameter will be used, or the {@link #DEFAULT_PORT} if no
		configuration parameter is found.
*/
public Dispatcher
	(
	String			name,
	Configuration	configuration,
	int				port
	)
{
The_Configuration = configuration;
Port = port;
Dispatcher_Name (name);
}

/**	Construct a Dispatcher with a configuration and a specified
	communication port.
<p>
	The {@link #Dispatcher_Name() Dispatcher name} will be the {@link
	#DEFAULT_DISPATCHER_NAME}.
<p>
	<b.N.B.</b>: Constructing a Dispatcher does not {@link #Start() start}
	it operating.
<p>
	@param	configuration	The Configuration to use to {@link
		#Configure(Configuration) configure} the Dispatcher when it is
		{@link #Start() started}.
	@param	port	The communications port number. If less than or equal
		to zero the value of the configuration {@link #PORT_PARAMETER_NAME}
		parameter will be used, or the {@link #DEFAULT_PORT} if no
		configuration parameter is found.
*/
public Dispatcher
	(
	Configuration	configuration,
	int				port
	)
{this (null, configuration, port);}


/**	Construct a named Dispatcher with a configuration.
<p>
	The communication port number will be obtained from the configuration
	{@link #PORT_PARAMETER_NAME} parameter, or the {@link #DEFAULT_PORT}
	will be used if no configuration parameter is found.
<p>
	<b.N.B.</b>: Constructing a Dispatcher does not {@link #Start() start}
	it operating.
<p>
	@param	name	The {@link #Dispatcher_Name(String) Dispatcher name}.
	@param	configuration	The Configuration to use to {@link
		#Configure(Configuration) configure} the Dispatcher when it is
		{@link #Start() started}.
*/
public Dispatcher
	(
	String			name,
	Configuration	configuration
	)
{this (name, configuration, 0);}


/**	Construct a Dispatcher with a configuration.
<p>
	<b.N.B.</b>: Constructing a Dispatcher does not {@link #Start() start}
	it operating.
<p>
	The {@link #Dispatcher_Name() Dispatcher name} will be the {@link
	#DEFAULT_DISPATCHER_NAME}.
<p>
	The communication port number will be obtained from the configuration
	{@link #PORT_PARAMETER_NAME} parameter, or the {@link #DEFAULT_PORT}
	will be used if no configuration parameter is found.
<p>
	@param	configuration	The Configuration to use to {@link
		#Configure(Configuration) configure} the Dispatcher when it is
		{@link #Start() started}.
*/
public Dispatcher
	(
	Configuration	configuration
	)
{this (null, configuration, 0);}

/**	Construct a named Dispatcher.
<p>
	<b.N.B.</b>: Constructing a Dispatcher does not {@link #Start() start}
	it operating.
<p>
	When the Dispatcher is {@link #Configure(Configuration) configured}
	a default configuration source will be sought.
<p>
	The communication port number will be obtained from the configuration
	{@link #PORT_PARAMETER_NAME} parameter, or the {@link #DEFAULT_PORT}
	will be used if no configuration parameter is found.
<p>
	@param	name	The {@link #Dispatcher_Name(String) Dispatcher name}.
*/
public Dispatcher
	(
	String			name
	)
{this (name, null, 0);}

/**	Construct a Dispatcher.
<p>
	<b.N.B.</b>: Constructing a Dispatcher does not {@link #Start() start}
	it operating.
<p>
	The {@link #Dispatcher_Name() Dispatcher name} will be the {@link
	#DEFAULT_DISPATCHER_NAME}.
<p>
	When the Dispatcher is {@link #Configure(Configuration) configured}
	a default configuration source will be sought.
<p>
	The communication port number will be obtained from the configuration
	{@link #PORT_PARAMETER_NAME} parameter, or the {@link #DEFAULT_PORT}
	will be used if no configuration parameter is found.
*/
public Dispatcher ()
{this (null, null, 0);}

/*==============================================================================
	Accessors
*/
/**	Get the default Dispatcher name.
<p>
	@return	The default Dispatcher name.
	@see	#Default_Dispatcher_Name(String)
*/
public static String Default_Dispatcher_Name ()
{return Default_Dispatcher_Name;}

/**	Set the default Dispatcher name.
<p>
	@param	name	The default Dispatcher name. If null, the
		{@link #DEFAULT_DISPATCHER_NAME} is used.
*/
public static void Default_Dispatcher_Name
	(
	String	name
	)
{
if (name == null)
	name = DEFAULT_DISPATCHER_NAME;
Default_Dispatcher_Name = name;
}

/**	Get the Dispatcher name.
<p>
	@return	The Dispatcher name.
	@see	#Dispatcher_Name(String)
*/
public String Dispatcher_Name ()
{return Dispatcher_Name;}

/**	Set the Dispatcher name.
<p>
	The {@link #NAME_PARAMETER_NAME} in the Dispatcher {@link
	#Identity() identity} is set to the Dispatcher name.
<p>
	@param	name	The Dispatcher name. If null, the {@link
		#Default_Dispatcher_Name() default Dispatcher name} is used.
	@return	This Dispatcher.
*/
public synchronized Dispatcher Dispatcher_Name
	(
	String	name
	)
{
if (name == null)
	name = Default_Dispatcher_Name;
Dispatcher_Name = name;

//	Reset the Identity name.
if (Identity == null)
	{
	try {Identity (null);}
	catch (PVL_Exception exception) {}
	}
Identity.Set (NAME_PARAMETER_NAME, Dispatcher_Name);
return this;
}

/**	Get the Dispatcher communication port number.
<p>
	@return	The communication port number for this Dispatcher.
*/
public int Port ()
{return Port;}

/**	Get the Dispatcher Hello broadcast port number.
<p>
	@return	The Hello broadcast port number for this Dispatcher.
*/
public int Hello_Port ()
{return Hello_Port;}

/**	Get the Dispatcher Hello broadcast multicast address.
<p>
	@return	The Hello broadcast multicast address String for this Dispatcher.
*/
public String Hello_Address ()
{return Hello_Address;}

/**	Get the Dispatcher location.
<p>
	The format of the location string is:
<p>
	<i>host</i>[<b><i>:</i></b><i>port</i>]
<p>
	The <i>host</i> may be a hostname or IP address. The hostname is
	{@link Host#Full_Hostname() fully qualified}. The communications
	{@link #Port() port number} is appended after a colon (':')
	delimiter.
<p>
	@return	A String specifying the location of this Dispatcher.
*/
public String Location ()
{return Host.Full_Hostname () + ':' + Port;}

/**	Get the Dispatcher identity.
<p>
	@return	A Message containing the Dispatcher identification information.
		This will be null if, and only if, the identity Message could not
		be assembled.
	@see	#Identity(Message)
*/
public Message Identity ()
{
try {return new Message (Identity);}
catch (PVL_Exception exception)
	{
	//	The Identity Message has been corrupted!
	try 
		{
		Identity (null);
		return new Message (Identity);
		}
	catch (PVL_Exception except) {/* Shouldn't happen */}
	}
return null;
}

/**	Set the Dispatcher identity.
<p>
	The following parameters will always be set in the identity Message:
<p><dl>
<dt>{@link #ACTION_PARAMETER_NAME}
<dd>Set to {@link #IDENTITY_ACTION}.

<dt>{@link #NAME_PARAMETER_NAME}
<dd>Set to the {@link #Dispatcher_Name() Dispatcher name}.

<dt>{@link #CLASS_ID_PARAMETER_NAME}
<dd>Set to the {@link #Dispatcher_ID}.

<dt>{@link Configuration#HOST}
<dd>Set to the {@link Host#FULL_HOSTNAME}.

<dt>{@link Configuration#USER}
<dd>Set to the account name under which the Dispatcher is running.
</dl>
<p>
	@param	identity	A Message containing the Dispatcher
		identification information. If null an empty Message is provided
		before setting the required parameters.
	@return	This Dispatcher.
	@throws	PVL_Exception if the identity Message can not be assembled
		either because the specified identity Message can not be copied
		or the required parameters can not be set.
*/
protected synchronized Dispatcher Identity
	(
	Message	identity
	)
	throws PVL_Exception
{
if (identity == null)
	Identity = new Message ();
else
	Identity = new Message (identity);
Identity
	.Set (ACTION_PARAMETER_NAME, IDENTITY_ACTION)
	.Set (NAME_PARAMETER_NAME, Dispatcher_Name)
	.Set (CLASS_ID_PARAMETER_NAME, Dispatcher_ID)
	.Set (Configuration.HOST, Host.FULL_HOSTNAME)
	.Set (Configuration.USER, System.getProperty ("user.name"));
return this;
}

/**	Get the maximum time to wait for a response to a new client identify
	request.
<p>
	@return	The identify reponse timeout (seconds).
	@see	#Identify_Timeout(int)
*/
public int Identify_Timeout ()
{return Identify_Timeout;}

/**	Set the maximum time to wait for a response to a new client identify
	request.
<p>
	If a new client connection does not respond to the identity request
	before the timeout expires the client connection is dropped.
<p>
	@param	seconds	The identify reponse timeout (seconds). If less than
		or equal to zero the DEFAULT_IDENTIFY_TIMEOUT will be used.
	@return	This Dispatcher.
	@see	#Identify_Timeout(int)
*/
public Dispatcher Identify_Timeout
	(
	int		seconds
	)
{
if (seconds < MINIMUM_IDENTIFY_TIMEOUT)
	seconds = MINIMUM_IDENTIFY_TIMEOUT;
Identify_Timeout = seconds;
return this;
}

/**	Enable or disable automatic reporting of the current Messagers
	list whenever it changes.
<p>
	Whenever a client connects or disconnects a report is sent to all
	Messengers that have {@link #Start_Messenger_Reporting(Messenger)
	requested connection reports} that provides the identities of all
	Messengers in the modified list of connected Messagers. For Message
	protocol operations that may generate a flurry of connection events
	it can be desirable to suspend automatic connection event reporting
	until after the protocol operation has been completed. This can
	significantly reduce the amount of connection list change processing
	that clients are required to manage.
<p>
	@param	enabled	If true whenever the list of connected Messagers
		changes it will be reported to all Messengers that have requested
		connection reports; if false reports will only be provided on
		demand.
	@return	This Dispatcher.
	@see	#Report_Messengers()
*/
public Dispatcher Report_Messenger_Connections
	(
	boolean	enabled
	)
{
Report_Messenger_Connections = enabled;
return this;
}

/**	Test if automatic reporting of the current Messagers list is enabled.
<p>
	@return	true if automatic reporting of the current Messagers list is
		enabled; false otherwise.
	@see	#Report_Messenger_Connections(boolean)
*/
public boolean Report_Messenger_Connections ()
{return Report_Messenger_Connections;}

/**	Get any Exception that might have occured when the Dispatcher was used
	as a {@link #run() Runnable}.
<p>
	When used as a Runnable the Dispatcher connection listener must first be
	{@link #Start() started}. If this throws an exception it is held since
	it can not be thrown from a Runnable.
<p>
	@return	The Exception that was thrown when the Dispatcher connection
		listener was {@link #Start() started} as a result of {@link #run()
		running} the Dispatcher. This will be null if no exception was
		thrown.
*/
public Exception Run_Exception ()
{return Run_Exception;}

/*==============================================================================
	Configuration
*/
/**	Get the default configuration source.
<p>
	The default configuration source is tried if no Configuration is
	supplied to {@link #Configure(Configuration) configure} the Dispatcher
	and a configuration source based on the {@link #Dispatcher_Name()
	Dispatcher name} can not be found.
<p>
	@return	The name of the default source file. This is the {@link
		#Default_Dispatcher_Name() default Dispatcher name} plus the
		".conf" extension.
*/
public static String Default_Configuration_Source ()
{return Default_Dispatcher_Name + ".conf";}

/**	Configures the Dispatcher.
<p>
	If no Configuration is provided a search will be made for a default
	configuration source. The {@link #Dispatcher_Name() Dispatcher name}
	plus the ".conf" extension will be tried first. If this fails, the
	{@link #Default_Configuration_Source() default configuration source}
	will be tried.
<p>
	The Configuration is set to be {@link
	Configuration#Case_Sensitive(boolean) case insensitive} before
	parameters are sought.
<p>
	Configuration parameters that may be used:
<dl>
<dt>{@link #PORT_PARAMETER_NAME}
	<dd>If the communications port number was not set when the Dispatcher
		was constructed the value of this parameter will be used. The
		default value is {@link #DEFAULT_PORT}.
<dt>{@link #IDENTIFY_TIMEOUT_PARAMETER_NAME}
	<dd>The number of seconds to wait for a Messenger client connection
		to respond to the identify message it is sent. The default value
		is {@link #DEFAULT_IDENTIFY_TIMEOUT}; the minium value is
		{@link #MINIMUM_IDENTIFY_TIMEOUT}.
<dt>{@link #PASSWORD_PARAMETER_NAME}
	<dd>The Messenger client authentication password. <b>N.B.</b>: If no
		password is provided, or it is emtpy, and {@link
		#UNAUTHENTICATED_CONNECTIONS_ALLOWED} is true unauthenticated
		connections will be allowed.
<dt>{@link #LOG_PATHNAME_PARAMETER_NAME}
	<dd>If {@link #Logging(boolean) logging} has not been enabled, and a
		log pathname is found, the pathname is used to set the {@link
		#Log_Pathname(String) log file} and logging is then enabled.
		<b>N.B.</b>: A configuration log pathname will not override an
		application set log file if logging has been enabled by the
		application.
<dt>{@link #HELLO_PORT_PARAMETER_NAME}
	<dd>After the Dispatcher starts and is ready to receive connections
		it will broadcast a {@link #HELLO_MESSAGE} on this port to
		notify any clients that may listening that they may open a
		connection. If the port number is zero no Hello broadcast will
		be sent. {@link #DEFAULT_HELLO_PORT} is the default port number.
		If the port number is not positive no broadcast will be done.
<dt>{@link #HELLO_ADDRESS_PARAMETER_NAME}
	<dd>This address is expected to be a private multicast address
		suitable for use in broadcasting the Hello message. The
		time-to-live of datagrams that are sent is zero to keep them on
		the local system. {@link #DEFAULT_HELLO_ADDRESS} is the default
		multicast address. If the address is empty no broadcast will be
		done.
</dl><p>
	<b>N.B.</b>: All parameters are sought in the {@link
	#Config_Pathname(String) in the Configuration Group} with the {@link
	#Dispatcher_Name() Dispatcher name}.
<p>
	@param	configuration	The Configuration to use. If null and
		defatult configuration source will sought.
	@throws	Configuration_Exception	If there was a problem parsing the
		configuration source, accessing its contents, or establishing
		a log file.
	@throws	SecurityException	If the connection authentication keys
		could not be generated, no {@link #PASSWORD_PARAMETER_NAME} was
		found but {@link #UNAUTHENTICATED_CONNECTIONS_ALLOWED} is false,
		or a log file could not be opened due to a security violation.
*/
protected void Configure
	(
	Configuration	configuration
	)
	throws	Configuration_Exception, SecurityException
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Configure");
if (configuration == null)
	{
	String
		source = Dispatcher_Name + ".conf";
	if ((DEBUG & DEBUG_CONFIG) != 0)
		System.out.println
			("    Trying " + source);
	try {configuration = new Configuration (source);}
	catch (IllegalArgumentException exception) {}
	if (configuration == null &&
		! Dispatcher_Name.equals (Default_Dispatcher_Name))
		{
		//	Fallback to the default name.
		source = Default_Dispatcher_Name + ".conf";
		if ((DEBUG & DEBUG_CONFIG) != 0)
			System.out.println
				("    Trying " + source);
		try {configuration = new Configuration (source);}
		catch (IllegalArgumentException exception) {}
		}
	if (configuration == null)
		throw new Configuration_Exception (ID + NL
			+ "Unable to find the default configuration file \""
				+ Dispatcher_Name + ".conf\""
				+ ((! Dispatcher_Name.equals (Default_Dispatcher_Name)) ?
					(" nor \"" + Default_Dispatcher_Name + ".conf\".") : "."));
	}

The_Configuration = configuration;
The_Configuration.Case_Sensitive (false);
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("    Configuration -" + NL
		+ The_Configuration.Description ());

if (Port <= 0)
	Port = (int)The_Configuration.Get_Number
		(Config_Pathname (PORT_PARAMETER_NAME),
			DEFAULT_PORT);

Identify_Timeout ((int)The_Configuration.Get_Number
	(Config_Pathname (IDENTIFY_TIMEOUT_PARAMETER_NAME),
		DEFAULT_IDENTIFY_TIMEOUT));

//	Authentication password.
Password = The_Configuration.Get_One
	(Config_Pathname (PASSWORD_PARAMETER_NAME));
if (Password != null &&
	Password.length () == 0)
	Password = null;
if (Password != null)
	{
	if ((DEBUG & DEBUG_CONFIG) != 0)
		System.out.println
			("    Authentication configuration with password");
	try
		{
		Keys		= Authentication.Keys ();
		Public_Key	= Authentication.Public_Key (Keys);
		}
	catch (Exception exception)
		{
		throw (SecurityException)new SecurityException (ID + NL
			+ "Failed to initialize the authentication keys.")
			.initCause (exception);
		}
	}
else if (! UNAUTHENTICATED_CONNECTIONS_ALLOWED)
	throw new SecurityException (ID + NL
		+ "Unauthenticated connections are not allowed" + NL
		+ "and no " + PASSWORD_PARAMETER_NAME
			+ " parameter was found in the configuration file -" + NL
		+ The_Configuration.Source ());

if (! Logging)
	{
	String
		log_file = The_Configuration.Get_One
			(Config_Pathname (LOG_PATHNAME_PARAMETER_NAME));
	if (log_file != null)
		{
		try {Log_Pathname (log_file);}
		catch (FileNotFoundException exception)
			{
			throw new Configuration_Exception (ID + NL
				+ "Configuration "
					+ Config_Pathname (LOG_PATHNAME_PARAMETER_NAME)
					+ " parameter." + NL
				+ exception.getMessage ());
			}
		Logging (true);
		}
	}

Hello_Port = (int)The_Configuration.Get_Number
	(Config_Pathname (HELLO_PORT_PARAMETER_NAME),
		DEFAULT_HELLO_PORT);

Hello_Address = The_Configuration.Get_One
	(Config_Pathname (HELLO_ADDRESS_PARAMETER_NAME));
if (Hello_Address == null)
	Hello_Address = DEFAULT_HELLO_ADDRESS;
if (Hello_Address.length () == 0)
	Hello_Address = null;

if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("<<< Configure");
}

/**	Get an application specifiic configuration parameter pathname.
<p>
	If the specified name is null or an {@link
	Configuration#Is_Absolute_Pathname(String) absolute pathname} it is
	returned unmodified. Otherwise the name is added to the absolute
	pathname of the {@link #Dispatcher_Name() Dispatcher name} Group.
<p>
	@param	name	A parameter name. May be null.
	@return	An absolute Configuration parameter pathname (or null if the
		name is null).
*/
protected String Config_Pathname
	(
	String	name
	)
{
if (name != null &&
	! Configuration.Is_Absolute_Pathname (name))
	return
		The_Configuration.Path_Delimiter () + Dispatcher_Name +
		The_Configuration.Path_Delimiter () + name;
return name;
}

/*==============================================================================
	Logging
*/
/**	Get the default log filename.
<p>
	@return The default log filename. This will be the {@link
		#Default_Dispatcher_Name() default Dispatcher name} plus the
		".log" extension.
*/
public static String Default_Log_Filename ()
{return Default_Dispatcher_Name + ".log";}

/**	Get the log file pathname.
<p>
	@return	The log file pathname. This will be the last {@link
		#Log_Pathname(String) log pathname} that was set, or the
		{@link #Default_Log_Filename() default log filename} if
		a log pathname has not been set.
*/
public String Log_Pathname ()
{return Log_Pathname;}

/**	Set the pathname to the log file.
<p>
	If a {@link #Log_Stream() log stream} is already open it is flushed.
	If it is not the stdout, it is closed.
<p>
	If the pathname is {@link #STDOUT} the log stream is set to the
	{@link System#out stdout} stream. Otherwise the pathname is checked
	for an existing file. <b>N.B.</b>: An existing file is appended. If
	the file does not exist an attempt is made to create it. The
	{@link #Log_Stream() log stream} is put in auto-flush mode.
<p>
	@param	pathname	The pathname to the log file. If null or the
		empty String the {@link #Default_Log_Filename() default log
		filename} will be used.
	@return	A String that reports how the log file was established.
	@throws	FileNotFoundException	If an existing file could not be
		appended, possibly because it is not a regular file, or the
		file could not be created for writing.
	@see	#Log_Stream()
*/
public synchronized String Log_Pathname
	(
	String	pathname
	)
	throws FileNotFoundException
{
if (pathname == null ||
	pathname.length () == 0)
	pathname = Default_Log_Filename ();
Log_Pathname = pathname;

if (Log_Stream != null)
	{
	//	Shutdown the log stream.
	Log_Stream.flush ();
	if (Log_Stream != System.out)
		Log_Stream.close ();
	Log_Stream = null;
	}

if (Log_Pathname.equals (STDOUT))
	{
	Log_Stream = System.out;
	return "Logging to standard output.";
	}

File
	log_file = new File (Log_Pathname);
log_file = log_file.getAbsoluteFile ();
String
	report = null;
if (log_file.exists ())
	{
	if (log_file.isFile ())
		{
		//	Append to existing file.
		try {Log_Stream = new PrintStream (new FileOutputStream
				(log_file, true) , true);}
		catch (FileNotFoundException exception)
			{
			throw new FileNotFoundException (ID + NL
				+"Unable to append to the log file: "
					+ log_file.getAbsolutePath () + NL
			+ exception.getMessage ());
			}
		}
	else
		{
		throw new FileNotFoundException (ID + NL
			+"The file " + log_file.getAbsolutePath () + NL
			+"exists but is not a normal file.");
		}
	report = "Appending to log file: " + log_file.getAbsolutePath ();
	}
else
	{
	try {Log_Stream = new PrintStream (new FileOutputStream
			(log_file) , true);}
	catch (FileNotFoundException exception)
		{
		throw new FileNotFoundException (ID + NL
			+"Unable to create the log file: "
				+ log_file.getAbsolutePath () + NL
			+ exception.getMessage ());
		}
	report = "New log file: " + log_file.getAbsolutePath ();
	}
return report;
}

/**	Get the log stream.
<p>
	@return	The PrintStream where logging will be written. This will
		be null if no {@link #Log_Pathname(String) log file} has been
		established.
	@see	#Log_Pathname(String)
*/
public PrintStream Log_Stream ()
{return Log_Stream;}

/**	Test if logging is enabled.
<p>
	@return	true if logging is enabled; false otherwise.
	@see	#Logging(boolean)
*/
public boolean Logging ()
{return Logging;}

/**	Enable or disable logging.
<p>
	If logging is being enabled from a disabled state, and no {@link
	#Log_Stream() log stream} is open, the current {@link #Log_Pathname()
	log pathname} is used to open a log stream.
<p>
	If logging is being disabled from an enabled state the log stream is
	flushed but the log stream remains open.
<p>
	<b>N.B.</b>: Logging may be enabled and disabled while the Dispatcher
	is in operation.
<p>
	@param	enabled	true if logging is to be enabled; false otherwise.
	@return	true if logging was enabled; false otherwise.
*/
public synchronized boolean Logging
	(
	boolean	enabled
	)
{
if (enabled)
	{
	if (! Logging)
		{
		//	Start logging.
		if (Log_Stream == null)
			{
			try
				{
				Log_Pathname (Log_Pathname);
				Logging = true;
				}
			catch (FileNotFoundException exception) {}
			}
		else
			Logging = true;
		}
	}
else if (Logging)
	{
	//	Stop logging.
	Logging = false;
	Log_Stream.flush ();
	}
return Logging;
}

/**	Writes a log report with a timestamp.
<p>
	If logging is disabled nothing is done.
<p>
	The current date and time is written using the {@link #DATE_FORMAT}
	on a line by itself. This is followed by the report text. If the
	report text is null a single empty line is written.
<p>
	@param	report	The report text to be logged. If null a single empty
		line is written.
	@see	#Logging(boolean)
	@see	#Log_Pathname(String)
*/
public void Log_Time
	(
	String	report
	)
{Log (new Date (), report);}

/**	Writes a log report.
<p>
	If logging is disabled nothing is done.
<p>
	@param	report	The report text to be logged. If null a single empty
		line is written.
	@see	#Logging(boolean)
	@see	#Log_Pathname(String)
*/
public void Log
	(
	String	report
	)
{Log (null, report);}

/**	Writes an empty log line.
<p>
	@see	#Log(String)
*/
public void Log ()
{Log (null, null);}


private void Log
	(
	Date	timestamp,
	String	report
	)
{
if (Logging)
	synchronized (Log_Stream)
		{
		if (timestamp != null)
			Log_Stream.println (NL +
				DATE_FORMAT.format (timestamp));
		if (report != null)
			Log_Stream.println (report);
		else
			Log_Stream.println ();
		}
}

/*==============================================================================
	Listener
*/
/**	Start the Dispatcher.
<p>
	If the Dispatcher has already been started nothing is done.
<p>
	The Dispatcher is first {@link #Configure(Configuration) configured}.
<p>
	If {@link #Logging(boolean) logging} has been enabled a startup
	message is logged.
<p>
	Then the communications socket is established.
<p>
	The connection listener is then {@link #run() run} which listens for
	and handles Messenger client connections.
<p>
	<b>N.B.</b>: Once the Dispatcher is started it will continue to run
	until interrupted, a fatal exception occurs, or the JVM exits.
<p>
	@throws	Configuration_Exception	If there was a problem configuring
		the Dispatcher.
	@throws	SecurityException	If, during configuration, the connection
		authentication keys could not be generated, no {@link
		#PASSWORD_PARAMETER_NAME} was found but {@link
		#UNAUTHENTICATED_CONNECTIONS_ALLOWED} is false, or a log file
		could not be opened due to a security violation. This exception
		will also be thrown if a security violation was encountered while
		establishing the communication channel.
	@throws	IOException	If there was a problem while establishing the
		communication channel.
*/
public void Start ()
	throws
		Configuration_Exception,
		SecurityException,
		IOException
{
if (Started ())
	return;

JRE.addShutdownHook (new Thread ()
	{public void run ()
	{Shutdown ();}});

Configure (The_Configuration);

Log (NL
	+ START_STOP_MARKER);
Log_Time (ID + NL
	+ (ID.equals (Dispatcher_ID) ? "" : (Dispatcher_ID + NL))
	+ Dispatcher_Name + " start on host "
		+ Host.Full_Hostname () + '/' + Host.IP_Address ());

Bind_to_Socket ();

if (Password == null)
	Log (">>> WARNING <<< Connection authentication is not enabled.");

run ();
}

/**	Tests if this Dispatcher has been started.
<p>
	@return	true	If this Dispatcher has started its connection listener;
		false otherwise.
	@see	#Start()
*/
public boolean Started ()
{return Listener != null;}

/**	Run the connection listener.
<p>
	<b>N.B.</b>: A check is first made that the Dispatcher connection
	listener has been {@link #Started() started} and, if it has not,
	it is told to {@link #Start() start}. This enables the Dispatcher
	to be used a Thread Runnable. However, starting the connection
	listener causes an exception to be thrown the Runnable will not run.
	In this case the {@link #Run_Exception () run exception} must be
	specifically retrieved.
<p>
	While the listener socket that was bound when the Dispatcher
	was constructed is open the thread continually listens for new
	connections. Each connection is a potential new client.
	Each connection is allocated a new Messenger that uses the
	socket as its communication channel and this Dispatcher as
	its {@link Messenger#Employer() employer}.
<p>
	Each client must complete a protocol handshake:
<dl>
<dt>{@link Message#Identify() Identify}
<dd>The handshake is intiated by the Dispatcher {@link #Send(Messenger,
	Message) sending} the client via the new Messenger an Identify
	request Message. This Message contains an {@link
	Message#ACTION_PARAMETER_NAME Action} parameter with the {@link
	Message#IDENTIFY_ACTION Identify} value. If authentication is
	required or the Dispatcher has been configured with password a
	{@link #KEY_PARAMETER_NAME Key} parameter will also be included that
	has an encoded public key value.

<dt>{@link #Authentication(Message, Message) Authentication}
<dd>The client on receiving the Identify request Message must use the
	public key it provides to generate authentication information that
	will be included with its other Identity information. This is done
	by setting the {@link #KEY_PARAMETER_NAME Key} parameter in a copy
	of its usual Identity Message to the required password value and
	then passing both the Identify and Identity Messages to the
	Authentication utility which will replace the password value of the
	Identity Key parameter with an encrypted and encoded value created
	using the public key from the Identify Message. If client
	authentication is not required no authentication information needs
	to be provided. The Identity Message is sent by the client in
	response to the Identify Message.

<dt>{@link Message#Identity() Identity}
<dd>The Dispatcher expects to {@link Messenger#Receive(int) receive} an
	Identity Message in response to its Identify request within the
	{@link #Identify_Timeout() timeout} period or the Messenger will be
	told that it is {@link Messenger#Done(String, Exception) done} which
	will drop the connenction. The identity Message and its route-from
	addressing is {@link #Log(String) logged}.

<dt>{@link #Authenticate(Message) Authenticate}
<dd>The client's Identity Message is used for authentication. If {@link
	#UNAUTHENTICATED_CONNECTIONS_ALLOWED} is true and the Dispatcher has
	not been {@link #Configure(Configuration) configured} with a
	password then client Identity authentication always succeeds. When
	password authentication is required for the Identity authentication
	to succeed the Identity Message must include a {@link
	#KEY_PARAMETER_NAME} parameter with a value that can be decoded and
	decrypted using the Dispatcher's private key to produce a password
	value that is identical to the one with which it has been
	configured.

<dt>Conclusion
<dd>If authentication of the client's Identity Message succeeded the
	Dispatcher will conclude the handshake by sending its own {@link
	#Identity() Identity} Message to the client. If authentication
	failed a {@link #NACK(Messenger, Message) NACK} Message will be sent
	containing the client's Identity without any Key parameter plus an
	{@link #EXPLANATION_PARAMETER_NAME Explanation} parameter saying
	that the Identity did not authenticate.
</dl>
	If any part of the handshake fails the Messenger is told that it is
	{@link Messenger#Done(String) done} which closes down the Messenger and
	closes its connection.
<p>
	After conclusion of the protcol handshake the route-to address list
	of the identity Message is set to its route-from list and the Message
	is bound the Messenger as its (@link Messenger#Identity(Message)
	identity}. The Messenger is added to the list of connected Messengers
	and is told to asynchronously listen for Messages. Then the list of
	{@link #Start_Messenger_Reporting(Messenger) Report_to_Messengers} are
	sent a Message that includes the {@link
	#Messenger_Identities(Messenger, String) identities of the connected
	Messengers}.
<p>
	Each connection step is logged.
<p>
	@see	SocketChannel
*/
public void run ()
{
if (! Started ())
	{
	try {Start ();}
	catch (Exception exception)
		{
		Run_Exception = exception;
		return;
		}
	}

SocketChannel
	client;
Messenger
	messenger;
Message
	message,
	identity;
Exception
	exception;
String
	report;

if (Hello_Port > 0 &&
	Hello_Address != null &&
	Hello == null)
	{
	Log ("Sending the \"" + HELLO_MESSAGE
			+ "\" broadcast to multicast address " + Hello_Address
			+ " via port " + Hello_Port + '.' + NL);
	Hello = new Say_Hello ();
	Hello.start ();
	}

Log ("Listening for client connections ..." + NL);

while (Listener.isOpen ())
	{
	client = null;
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("... Dispatcher: Listening for connection ...");
	try {client = Listener.accept ();}
	catch (IOException except)
		{
		if (except instanceof AsynchronousCloseException)
			//	The Listener has been closed.
			break;

		Log_Time ("Connection exception -" + NL
			+ except);
		}
	catch (SecurityException except)
		{
		Log_Time ("Client connection prohibited -" + NL
			+ except);
		}
	catch (Exception except)
		{
		Log_Time ("Unexpected connection exception:");
		if (Logging)
			except.printStackTrace (Log_Stream ());
		System.exit (EXIT_UNKNOWN_EXCEPTION);
		}
		
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("==> Dispatcher: Connection"
				+ ((client == null) ? " (no client)" : ""));
	if (client == null)
		continue;

	//	Assign the client a messenger.
	messenger = new Messenger (this, client);
	Log_Time ("Client connection: " + NL
		+ messenger);

	//	Ask the client to provide identification.
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("--> Dispatcher: Sending the client an "
				+ IDENTIFY_ACTION + " Message");
	/*
		The public key to be used for authentication is included.
		However, if authentication is not enabled there will be no
		public key and so the key parameter will not be set.
	*/
	if (! Send (messenger,
			Message.Identify ()
				.Set (KEY_PARAMETER_NAME, Public_Key)))
		{
		Log ("Unable to send to the client." + NL);
		messenger = null;
		continue;
		}

	//	Get the client identification.
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("--> Dispatcher: Receiving the client "
				+ IDENTIFY_ACTION + " Message");
	message = null;
	identity = null;
	exception = null;
	report = null;
	try
		{
		message = messenger.Receive (Identify_Timeout);
		identity = new Message (message)
					.Set (KEY_PARAMETER_NAME, null);
		}
	catch (IOException except)
		{
		exception = except;
		report =
			"The Stage_Manager did not receive an identity from the client.";
		}
	catch (PVL_Exception except)
		{
		exception = except;
		report =
			"Invalid identity message.";
		}
	catch (Exception except)
		{
		Log_Time ("Messenger exception:");
		if (Logging)
			except.printStackTrace (Log_Stream ());
		System.exit (EXIT_UNKNOWN_EXCEPTION);
		}
	if (message == null)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("--> Dispatcher: " + exception);
		Log (report + NL
			+ exception.getMessage () + NL);
		messenger.Done (report, exception);
		messenger = null;
		continue;
		}

	identity.Remove (ACTION_PARAMETER_NAME);
	Log (identity.Get (NAME_PARAMETER_NAME) + " client identity -" + NL
		+"Route from - " + identity.Route_From ().Description () + NL
		+ identity);
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("--> Dispatcher: Client identity -" + NL
			+"    Route from - " + identity.Route_From ().Description () + NL
			+ identity);

	//	Authenticate the client identification.
	if (! Authenticate (message))
		{
		report =
			"The Stage_Manager could not authenticate the client.";
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("--> Dispatcher: " + report);
		Log (report + NL);
		NACK (messenger,
			identity.Set (EXPLANATION_PARAMETER_NAME, report));
		messenger.Done (report);
		continue;
		}
	Log ();
	//	Send the server identity in response.
	Send (messenger, Identity ().Reply_To (identity));

	//	Move the Route_From list to the Route_To list.
	identity.Reply_To (identity);
	//	Apply the identification Message to the Messenger.
	messenger.Identity (identity);

	//	Add the Messenger to the connected list.
	synchronized (Messengers)
		{
		if (Messengers.contains (messenger))
			{
			//	This should never happen.
			if ((DEBUG & DEBUG_CONNECT) != 0)
				System.out.println
					("--> Dispatcher: Removing redundant Messenger connection!");
			Log
				("Multiple Messenger connection!" + NL
				+"Previous connection dropped:" + NL
				+ messenger);
			Messenger_Quit
				(messenger, "Multiple Messenger connection dropped.");
			}
		Messengers.add (messenger);
		}

	//	Start the Messenger listening to the client.
	messenger.Listen_for_Messages ();

	//	Notify interested clients of the new Messenger.
	if (Report_Messenger_Connections)
		Report_Messengers ();
	}

Log_Time ("Server socket closed.");
Listener = null;
}

/**	Disconnect a Messenger.
<p>
	The messenger is removed from the list of connected Messengers and
	the list of Messengers to receive reports of connected Messengers. If
	the Messenger was removed from the connected list, the disconnection
	is logged with the Messenger details and the report (if non-null),
	and the Messenger is told that it is {@link Messenger#Done(String)
	done} in case it has not already initiated its communication shutdown
	procedures.
<p>
	When the Messengers list is empty the shutdown-in-progress flag, that
	may have been set in the {@link #Shutdown() shutdown} procedure, is
	reset.
<p>
	@param	messenger	The Messenger to be disconnected.
	@param	report	A report to be logged. May be null.
*/
protected void Messenger_Quit
	(
	Messenger	messenger,
	String		report
	)
{
if ((DEBUG & DEBUG_MESSENGERS) != 0)
	System.out.println
		(">>> Dispatcher.Messenger_Quit:" + NL
		+ messenger + NL
		+ report);
boolean
	removed = false;
synchronized (Messengers)
	{
	removed = Messengers.remove (messenger);
	Report_to_Messengers.remove (messenger);
	}
if ((DEBUG & DEBUG_MESSENGERS) != 0)
	System.out.println
		("    The Messenger was " + (removed ? "" : "not ") + "removed");
if (removed)
	{
	Log_Time
		(messenger.Identity ().Get (NAME_PARAMETER_NAME)
			+ " Messenger disconnected:" + NL
		+ messenger + NL
		+"Route from - "
			+ messenger.Identity ().Route_To ().Description () + NL
		+ messenger.Identity ());
	if (report != null)
		Log (report);
	Log ();

	//	Make sure the Messenger is done.
	if (report == null)
		report = ID + NL;
	else
		report = ID + NL + report;

	messenger.Done (report);

	if (Report_Messenger_Connections)
		{
		//	Notify interested clients of the Messenger disconnection.
		if ((DEBUG & DEBUG_MESSENGERS) != 0)
			System.out.println
				("    Reporting new Messengers list");
		Report_Messengers ();
		}
	}

if (Messengers.size () == 0)
	Shutdown_In_Progress = false;

if ((DEBUG & DEBUG_MESSENGERS) != 0)
	System.out.println
		("<<< Dispatcher.Messenger_Quit");
}

/**	Disconnect a Messenger.
<p>
	If a message is specified and it contains an {@link
	#EXPLANATION_PARAMETER_NAME} and/or an {@link #EXCEPTION_PARAMETER_NAME}
	the values of those parameters are assembled into report to be included
	in the log. If there are any additional parameters, other than the
	{@link #ACTION_PARAMETER_NAME}, a description of all remaining parameters
	are also included in the log report.
<p>
	@param	messenger	The Messenger to be disconnected.
	@param	message	A Message to be logged. May be null.
	@see	#Messenger_Quit(Messenger, String)
*/
protected void Messenger_Quit
	(
	Messenger	messenger,
	Message		message
	)
{
String
	report = null;
if (message != null)
	{
	String
		string;
	string = Message.Unescape (message.Get (EXPLANATION_PARAMETER_NAME));
	if (string != null)
		{
		report = string;
		message.Remove (EXPLANATION_PARAMETER_NAME);
		}
	string = message.Get (EXCEPTION_PARAMETER_NAME);
	if (string != null &&
		! string.startsWith ("java.io.EOFException"))
		{
		if (report == null)
			report = "";
		else
			report += NL;
		report += Message.Unescape (string);
		}
	message.Remove (EXCEPTION_PARAMETER_NAME);
	message.Remove (ACTION_PARAMETER_NAME);
	if (message.List_Size () != 0)
		{
		if (report == null)
			report = "";
		else
			report += NL;
		report += message;
		}
	}
Messenger_Quit (messenger, report);
}


private volatile boolean
	Shutdown_In_Progress	= false;

/**	Shutdown the Dispatcher.
<p>
	The communications channel is closed. If there are any registered
	Messengers they are told they are {@link Messenger#Done(String)
	Done} and a shutdown-in-progress flag is set. This flag is checked
	up to five times with a two second delay between checks until either
	the flag is found to be reset or the maximum number of checks have
	been done. Finally a shutdown notice is logged that indicates
	whether the shutdown of all the Messegers was completed.
*/
protected void Shutdown ()
{
try {Listener.close ();}
catch (IOException exception) {}
try {Thread.sleep (1000);}
catch (InterruptedException exception) {}

int
	index = Messengers.size ();
if (index > 0)
	{
	Shutdown_In_Progress = true;
	Log_Time ("Shutdown of the " + Dispatcher_Name () + " initiated.");
	while (--index >= 0)
		Messengers.get (index).Done
			("The " + Dispatcher_Name () + " is shutting down.");
	}

index = 5;
while (index-- > 0)
	{
	if (Shutdown_In_Progress)
		{
		try {Thread.sleep (2000);}
		catch (InterruptedException exception) {}
		}
	}
Log_Time
	("Shutdown of the " + Dispatcher_Name () + " Messengers "
		+ ((Messengers.size () == 0) ? "is" : "did not fully")
		+ " complete." + NL
	+ NL
	+ START_STOP_MARKER);
}

/**	Get a Messenger for an address.
<p>
	The list of connected Messengers is searched for one that has an
	{@link Messenger#Address() address} that matches the one that is
	specified.
<p>
	@param	address	The address String associated with a Messenger.
		If null or the empty String null is returned.
	@return	The connected Messenger having the specified address, or
		null if no Messenger with that address was found.
*/
protected Messenger Lookup_Messenger
	(
	String	address
	)
{
if (address != null &&
	address.length () != 0)
	{
	synchronized (Messengers)
		{
		Messenger
			messenger;
		int
			index = 0,
			total = Messengers.size ();
		while (index < total)
			{
			messenger = Messengers.get (index++);
			if (address.equals (messenger.Address ()))
				return messenger;
			}
		}
	}
return null;
}

/*------------------------------------------------------------------------------
	Communications Channel
*/
private void Bind_to_Socket ()
	throws IOException, SecurityException
{
if (Listener != null &&
	Listener.isOpen ())
	{
	try {Listener.close ();}
	catch (IOException exception)
		{
		Log_Time ("Problem closing the server socket:" + NL
			+ exception.getMessage () + NL);
		}
	}
Log ("Initializing the server socket on port " + Port + '.');
try
	{
	if ((DEBUG & DEBUG_BIND) != 0)
		System.out.println
			("    Opening ServerSocketChannel");
	Listener = ServerSocketChannel.open ();
	if ((DEBUG & DEBUG_BIND) != 0)
		System.out.println
			("    Configure for blocking");
	Listener.configureBlocking (true);
	if ((DEBUG & DEBUG_BIND) != 0)
		System.out.println
			("    Setting socket reuse address");
	Listener.socket ().setReuseAddress (true);
	if ((DEBUG & DEBUG_BIND) != 0)
		System.out.println
			("    Binding");
	Listener.socket ().bind (new InetSocketAddress (Port));
	}
catch (IOException exception)
	{
	throw new IOException (ID + NL
		+ "Unable to construct a ServerSocketChannel on port " + Port + '.' + NL
		+ exception.getMessage ());
	}
catch (SecurityException exception)
	{
	throw new SecurityException (ID + NL
		+ "Binding a ServerSocket to port " + Port + " is prohibited." + NL
		+ exception.getMessage ());
	}
}

/*==============================================================================
	Authentication
*/
/**	Provide client identity authentication information.
<p>
	During the handshake protocol, while establishing a conneciton to a
	Dispatcher, authentication information will need to be exchanged
	unless {@link #UNAUTHENTICATED_CONNECTIONS_ALLOWED}. In the {@link
	#IDENTIFY_ACTION} Message that the Dispatcher sends to initiate the
	handshake for a new client connection a {@link #KEY_PARAMETER_NAME}
	parameter will provide a public key value that must be used to
	produce an encoded password value returned in the {@link
	#KEY_PARAMETER_NAME} parameter of the required {@link
	#IDENTITY_ACTION} Message response.
<p>
	The authentication key in the Identify Message is used with the
	authentication key in the Identity Message to produce an encoded
	authentication key in the Identity Message. If either the Identify or
	Identity Message do not have an authentication key then the returned
	Message will not have this parameter. The mechanism by which the
	Identify key is used to encode the Identity key is opaque.
<p>
	<b>N.B.</b>: The Identity Message that is provided is modified,
	not copied.
<p>
	@param	identify	A Message from which to obtain a {@link
		#KEY_PARAMETER_NAME} parameter with a public key authentication
		value. If null the identity Message is returned with any {@link
		#KEY_PARAMETER_NAME} parameter removed.
	@param	identity	A Message from which to obtain a {@link
		#KEY_PARAMETER_NAME} parameter with a private password authentication
		value. The value is replaced with encoded authentication
		information that combines the public key with the password. If null
		null is returned.
	@return	The possibly modified identity Message. This will have a
		modified {@link #KEY_PARAMETER_NAME} parameter if, and only if,
		the original value was successfully encoded into the appropriate
		authentication response value.
	@see	Authentication
*/
public static Message Authentication
	(
	Message identify,
	Message identity
	)
{
if (identify == null)
	identity.Set (KEY_PARAMETER_NAME, null);
else
if (identity != null)
	{
	if ((DEBUG & DEBUG_AUTHENTICATE) != 0)
		System.out.println
			(">-< Authentication: " + identity.Get (NAME_PARAMETER_NAME));
	String
		key = identify.Get (KEY_PARAMETER_NAME),
		password = identity.Get (KEY_PARAMETER_NAME);
	identity.Set
		(KEY_PARAMETER_NAME, Authentication.Encoded_Password (password, key));
	}
return identity;
}

/**	Authenticate a client identity.
<p>
	If no password was obtained from the {@link #Configure(Configuration)
	configuration} and {@link #UNAUTHENTICATED_CONNECTIONS_ALLOWED} is
	true, the identity is accepted without further authentication.
<p>
	Authentication requires that the identity contain a {@link
	#KEY_PARAMETER_NAME} parameter value that is used as the {@link
	#Authentication(Message, Message) encoded authentication password}
	to {@link Authentication#Authenticate(KeyPair, String, String)
	authenticate} the client identity. The decoded authentication
	password from the identity, after being decrypted using the
	Dispatcher's private key, must match the Dispatcher password.
<p>
	@return	true if the identity is authenticated or unauthenticated
		connections are allowed. false if the Dispatcher does not have a
		password and unauthenticated connections are not allowed, the
		identity does not contain a Key parameter, or its decoded and
		decrypted value does not match the password.
	@see	Authentication
*/
protected boolean Authenticate
	(
	Message		identity
	)
{
if ((DEBUG & DEBUG_AUTHENTICATE) != 0)
	System.out.println
		(">>> Authenticate: " + NL
		+ identity);
if (Password == null)
	{
	//	Unauthenticated connection.
	if ((DEBUG & DEBUG_AUTHENTICATE) != 0)
		System.out.println
			("<<< Authenticate: unauthenticated connection");
	return UNAUTHENTICATED_CONNECTIONS_ALLOWED;
	}

boolean
	authenticated =
		Authentication.Authenticate
			(Keys, identity.Get (KEY_PARAMETER_NAME), Password);
if ((DEBUG & DEBUG_AUTHENTICATE) != 0)
	System.out.println
		("<<< Authenticate: " + authenticated);
return authenticated;
}

/*==============================================================================
	Messages
*/
//	Message_Delivered_Listener
/**	Get the identity of this Dispatcher.
<p>
	This method implements the {@link Message_Delivered_Listener} interface.
<p>
	@return	A Message containing the Dispatcher {@link #Identity(Message)
		identity} information.
	@see	#Identity()
*/
public Message Listener_Identity ()
{return Identity ();}

//	Message_Delivered_Listener
/**	Receives delivery of a message.
<p>
	This method implements the {@link Message_Delivered_Listener} interface.
<p>
	The following {@link Message#Action() message actions} are recognized:
<dl>
<dt>{@link #IDENTIFY_ACTION}
	<dd>Replies with the {@link #Listener_Identity() identity Message}.
<dt>{@link #MESSENGERS_REPORT_ACTION}
	<dd>Replies with a list of {@link #Messenger_Identities(Messenger,
		String) currently connected messenger identities} that have been
		selected by the request message.
<dt>{@link #START_MESSENGER_REPORTING_ACTION}
	<dd>{@link #Start_Messenger_Reporting(Messenger) Starts connected
		Messenger reporting} for the requesting Messenger.
<dt>{@link #STOP_MESSENGER_REPORTING_ACTION}
	<dd>{@link #Stop_Messenger_Reporting(Messenger) Stops connected
		Messenger reporting} for the requesting Messenger.
<dt>{@link #LINK_MESSENGER_ACTION}
	<dd>Direct {@link #Messenger_Link(Messenger, Message)}
		Messenger-to-Messenger communication} is established between the
		requesting Messenger and another Messenger for which an address
		is provided.
<dt>{@link #UNLINK_MESSENGER_ACTION}
	<dd>Direct {@link #Messenger_Unlink(Messenger, Message)}
		Messenger-to-Messenger communication} is removed between the
		requesting Messenger and another Messenger for which an address
		is provided.
<dt>{@link #DONE_ACTION}
	<dd>The Messenger associated with this Message has
		{@link #Messenger_Quit(Messenger, Message) quit} and is to be
		disconnected.
<dt>{@link #UNDELIVERABLE_ACTION}
	<dd>A Message that was {@link #Send(Messenger, Message) sent} could
		not be delivered. The undeliverable Message is logged.
<dt>{@link #NACK_ACTION}
	<dd>A Message that was {@link #Send(Messenger, Message) sent} was
		rejected. The rejected Message is logged.
</dl><p>
	Any other messages are logged as unrecognized and a {@link
	#NACK(Messenger, Message) NACK} is sent back to the Messenger from
	which the message was received.
<p>
	@param	event	The {@link Message_Delivered_Event} containing the
		Message that was delivered and a reference to the Messenger that
		sent it.
*/
public void Message_Delivered
	(
	Message_Delivered_Event	event
	)
{
Messenger
	messenger = (Messenger)event.getSource ();
String
	action = event.Message.Action ();
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Dispatcher.Message_Delivered:" + NL
		+"    From " + messenger.Identity ().Get (NAME_PARAMETER_NAME) + " -" + NL
		+ messenger + NL
		+ event.Message.Routing () + NL
		+ event.Message);
try
{
if (IDENTIFY_ACTION.equals (action))
	Send (messenger,
		Listener_Identity ()
		.Reply_To (event.Message));
else
if (MESSENGERS_REPORT_ACTION.equals (action))
	Send (messenger,
		Messenger_Identities
			(messenger, event.Message.Get (NAME_PARAMETER_NAME))
		.Reply_To (event.Message));
else
if (START_MESSENGER_REPORTING_ACTION.equals (action))
	Start_Messenger_Reporting (messenger);
else
if (STOP_MESSENGER_REPORTING_ACTION.equals (action))
	Stop_Messenger_Reporting (messenger);
else
if (LINK_MESSENGER_ACTION.equals (action))
	Messenger_Link (messenger, event.Message);
else
if (UNLINK_MESSENGER_ACTION.equals (action))
	Messenger_Unlink (messenger, event.Message);
else
if (STATUS_REPORT_ACTION.equals (action))
	Status_Report (messenger, event.Message);
else
if (DONE_ACTION.equals (action))
	Messenger_Quit (messenger, event.Message);
else
if (UNDELIVERABLE_ACTION.equals (action))
	Log_Time ("Undeliverable message from "
			+ messenger.Identity ().Get (NAME_PARAMETER_NAME) + " -" + NL
		+ messenger + NL
		+ event.Message);
else
if (NACK_ACTION.equals (action))
	Log_Time ("Rejected message from "
			+ messenger.Identity ().Get (NAME_PARAMETER_NAME) + " -" + NL
		+ messenger + NL
		+ event.Message);
else
	{
	Log_Time ("Unrecognized message from "
			+ messenger.Identity ().Get (NAME_PARAMETER_NAME) + " -" + NL
		+ messenger + NL
		+ event.Message);
	NACK (messenger, event.Message);
	}
}
catch (Exception exception)
	{
	if (Logging)
		{
		Log_Time ("An exception occured while delivering a message -" + NL
			+ messenger + NL
			+ event.Message.Routing () + NL
			+ event.Message);
		exception.printStackTrace (Log_Stream ());
		}
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Dispatcher.Message_Delivered");
}

/*==============================================================================
	Message handlers
*/
/**	Send a Message via a Messenger.
<p>
	If failure reporting is enabled and the Messenger throws an exception
	while sending the Message this is logged with the Messenger
	description, the Message routing addresses, the Message description
	and the exception description. In any case if the exception is an
	IOException the Messenger is {@link #Messenger_Quit(Messenger,
	String) quit}.
<p>
	@param	messenger	The Messenger used to send the Message. If null
		nothing is done.
	@param	message		The Message to be sent. If null nothing is done.
	@param	report_failure	true if failure reporting is enabled; false
		otherwise.
	@return	true if the Message was successfully sent; false otherwise.
*/
protected boolean Send
	(
	Messenger	messenger,
	Message		message,
	boolean		report_failure
	)
{
if (messenger == null ||
	message == null)
	return false;
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Dispatcher.Send: To " + messenger.Address () + NL
		+ messenger + NL
		+ message.Routing () + NL
		+ message);
boolean
	sent = true;
if (messenger.Is_Connected ())
	{
	try {messenger.Send (message);}
	catch (Exception exception)
		{
		if (report_failure)
			Log_Time
				("Unable to send a message." + NL
				+ messenger + NL
				+ message.Routing () + NL
				+ message + NL
				+ exception + NL);
		if (exception instanceof IOException)
			Messenger_Quit (messenger, exception.toString ());
		sent = false;
		}
	}
else
	sent = false;
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Dispatcher.Send: " + sent);
return sent;
}

/**	Send a Message via a Messenger.
<p>
	The Message is {@link #Send(Messenger, Message, boolean) sent} with
	failure reporting enabled.
<p>
	@param	messenger	The Messenger used to send the Message. If null
		nothing is done.
	@param	message		The Message to be sent. If null nothing is done.
	@return	true if the Message was successfully sent; false otherwise.
	@see	#Send(Messenger, Message, boolean)
*/
protected boolean Send
	(
	Messenger	messenger,
	Message		message
	)
{return Send (messenger, message, true);}

/**	Assemble a Message that includes the identities of selected
	connected Messengers.
<p>
	A connected Messenger is the client Messenger that initiated a
	connection, completed the identification Message handshake with the
	Dispatcher Messenger allocated to the client, and that is
	currently active (in the Messengers). Connected Messengers may be
	selected by {@link #NAME_PARAMETER_NAME} value. If no name is specified
	all connected Messengers are selected. However, the source Messenger
	to whom the Message will be sent is never selected.
<p>
	The assembled Message includes an {@link #IDENTITY_ACTION} group for
	each selected Messenger. The parameters in this group are the {@link
	Messenger#Identity() identity} Message parameters bound to the
	Messenger. In addition, a {@link #ROUTE_TO_PARAMETER_NAME} is added to
	each identity that is an array of Messenger {@link
	Messenger#Address() addresses} for each Messenger on the route from
	the source Messenger to the connected Messenger. The Value of this
	parameter can be used to set the {@link Message#Route_To(Value)
	route-to list} of a Message to be sent from the source Messenger to
	the destination connected Messenger.
<p>
	Since the route-to list of each connected Messenger {@link
	Messenger#Identity() identity Message} is in Dispatcher-to-client
	order the route-to list of the source Messenger must be reversed to
	produce a correct route-to address list in source-to-client order.
	This reversed address list is then completed with the route-to list
	of the client Messenger being identified. <b>N.B.</b>: Message
	route-to address lists are in LIFO order: the last address of the
	list is for the first Messenger on the transmission route, and the
	first address is for the destination client Messenger.
<p>
	@param	source_messenger	The Messenger to whom the returned
		Message is intended. Route-to address lists for each identified
		Messenger will be relative to this Messenger. If null, or the
		Messenger has no {@link Messenger#Identity() identity}, or the
		identity has no {@link Message#Route_To () route-to list} the
		returned Message will only contain an {@link
		#EXPLANATION_PARAMETER_NAME} parameter with a brief message value
		describing the problem.
	@param	name	The {@link #NAME_PARAMETER_NAME} value used to select
		connected Messengers. If null, all Messengers will be selected.
		The source_messenger is never selected.
	@return	A Message containing the identity and route-to address list
		of each selected Messenger. However, if source_messenger is null,
		has no identity, or the identity has no route-to list the Message
		will only contain an {@link #EXPLANATION_PARAMETER_NAME}
		parameter with a brief message value describing the problem.
*/
public Message Messenger_Identities
	(
	Messenger	source_messenger,
	String		name
	)
{
if ((DEBUG & (DEBUG_IDENTITIES | DEBUG_MESSENGERS)) != 0)
	System.out.println
		(">>> Dispatcher.Messenger_Identities:" + NL
		+"    For source Messenger -" + NL
		+ source_messenger + NL
		+"    Selecting name - " + name);
Message
	message = Message.Action (MESSENGERS_REPORT_ACTION);

if (source_messenger == null)
	{
	if ((DEBUG & (DEBUG_IDENTITIES | DEBUG_MESSENGERS)) != 0)
		System.out.println
			("<<< Dispatcher.Messenger_Identities: No source messenger!");
	message.Set (EXPLANATION_PARAMETER_NAME,
		"No source messenger!");
	return message;
	}
Message
	identity = source_messenger.Identity ();
if (identity == null)
	{
	//	This should never happen.
	if ((DEBUG & (DEBUG_IDENTITIES | DEBUG_MESSENGERS)) != 0)
		System.out.println
			("<<< Dispatcher.Messenger_Identities: No source identity!");
	message.Set (EXPLANATION_PARAMETER_NAME,
		"The source messenger does not have an identity!");
	return message;
	}

/*	Destination Messenger routing.

	The identity Message route-to list is in Dispatcher to source
	Messenger order. This needs to be reversed so the route-to list sent
	to the source client will be in source Messenger to Dispatcher order;
	i.e. the route-to list sent to the client must be in the order
	appropriate for the client's use.
*/
Value
	source_addresses = identity.Route_To ();
if (source_addresses.Array_Size () == 0)
	{
	//	This should never happen.
	if ((DEBUG & (DEBUG_IDENTITIES | DEBUG_MESSENGERS)) != 0)
		System.out.println
			("<<< Dispatcher.Messenger_Identities: No source route-to list!");
	message.Set (EXPLANATION_PARAMETER_NAME,
		"The source messenger identity does not have a route-to list!");
	return message;
	}
try {Collections.reverse (source_addresses.Vector_Data ());}
catch (PVL_Exception exception) {/* Route_To returns a valid Array */}

synchronized (Messengers)
	{
	Messenger
		messenger;
	int
		index = -1,
		total = Messengers.size ();
	while (++index < total)
		{
		messenger = Messengers.get (index);
		//	Don't include the source Messenger.
		if (messenger != source_messenger)
			{
			identity = messenger.Identity ();
			if (name == null ||
				name.equals (identity.Get (NAME_PARAMETER_NAME)))
				{
				try
					{
					//	Copy the identity parameters.
					identity = new Message (identity);

					//	Add the source route-to list to the client list.
					identity.Set (ROUTE_TO_PARAMETER_NAME,
						identity.Route_To ().Add
							(source_addresses.Vector_Data ()), 0);

					//	Add the identity to the identities message.
					message.Add (IDENTITY_ACTION, identity);
					}
				catch (PVL_Exception exception) {}
				}
			}
		}
	}
if ((DEBUG & (DEBUG_IDENTITIES | DEBUG_MESSENGERS)) != 0)
	System.out.println
		("    Identities -" + NL
		+ message + NL
		+"<<< Dispatcher.Messenger_Identities");
return message;
}

/**	Sends a report of all the {@link #Messenger_Identities(Messenger,
	String) Messenger identities} to each of the Messengers that have
	{@link #Start_Messenger_Reporting(Messenger) requested connection
	reports}. 
*/
protected void Report_Messengers ()
{
synchronized (Messengers)
	{
	if ((DEBUG & (DEBUG_IDENTITIES | DEBUG_MESSENGERS)) != 0)
		System.out.println
			(">>> Dispatcher.Report_Messengers: to "
				+ Report_to_Messengers.size () + " reportees");
	Messenger
		messenger;
	int
		index = -1,
		total = Report_to_Messengers.size ();
	while (++index < total)
		{
		try {messenger = Report_to_Messengers.get (index);}
		catch (ArrayIndexOutOfBoundsException exception)
			{
			/*	Somehow (?!?) the size of Report_to_Messengers has changed.
				But the index is no longer in range so they have all been done.
			*/
			break;
			}
		try {Send (messenger, Messenger_Identities (messenger, null)
				.Route_To (Route_To (messenger)),
				false);}	//	No failure report.
		catch (PVL_Exception exception) {/* Shouldn't happen */}
		}
	}
if ((DEBUG & (DEBUG_IDENTITIES | DEBUG_MESSENGERS)) != 0)
	System.out.println
		("<<< Dispatcher.Report_Messengers");
}

/**	Start automatic Messenger connection reporting for a Messenger.
<p>
	The specified Messenger is added, if it is not already present,
	to the list of Messengers that will be sent the list of all
	connected {@link #Messenger_Identities(Messenger, String)
	Messenger identities} whenever a new Messenger connects or
	disconnects.
<p>
	The Messenger is sent the current connected Messengers list.
<p>
	@param	messenger	The Messenger to receive automatic Messenger
		connection reporting. If null, nothing is done.
	@see	#Stop_Messenger_Reporting(Messenger)
*/
protected void Start_Messenger_Reporting
	(
	Messenger	messenger
	)
{
if (messenger == null)
	return;
synchronized (Messengers)
	{
	if (! Report_to_Messengers.contains (messenger))
		Report_to_Messengers.add (messenger);
	}

//	Send the current list of connected Messengers.
try {Send (messenger, Messenger_Identities (messenger, null)
		.Route_To (Route_To (messenger)),
		false);}	//	No failure report.
catch (PVL_Exception exception) {/* Shouldn't happen */}
}

/**	Stop automatic Messenger connection reporting for a Messenger.
<p>
	The specified Messenger is removed, if it is present,
	from the list of Messengers that will be sent the list of all
	connected {@link #Messenger_Identities(Messenger, String)
	Messenger identities}.
<p>
	@param	messenger	The Messenger to receive automatic Messenger
		connection reporting. If null, nothing is done.
	@see	#Start_Messenger_Reporting(Messenger)
*/
protected void Stop_Messenger_Reporting
	(
	Messenger	messenger
	)
{
if (messenger == null)
	return;
synchronized (Messengers)
	{Report_to_Messengers.remove (messenger);}
}

/**	Link two Messengers for I/O forwarding.
<p>
	The specified Message must contain an {@link #ADDRESS_PARAMETER_NAME}
	value that specifies the address of {@link #Lookup_Messenger(String)
	known Messenger} to which the source Messenger is to be linked. The
	linked Messenger is {@link Messenger#Add_Forwarding(Messenger) added
	to the forwarding list} of the source Messenger and the source
	Messenger is added to the forwarding list of the linked Messenger.
<p>
	The specified Message is returned to the source Messenger with the
	addition of an {@link #ACK_ACTION} token and an {@link
	#IDENTITY_ACTION} group containing the {@link Messenger#Identity()
	identity} of the linked Messenger.
<p>
	If a Messenger with the address to link to is not known the reply
	Message contains a {@link #NACK_ACTION} token and an {@link
	#EXPLANATION_PARAMETER_NAME}, instead of a Messenger identity, that
	describes the reason that the link could not be done.
<p>
	<b>N.B.</b>: If the Message contains either an {@link #ACK_ACTION} or
	{@link #NACK_ACTION} token nothing is done.
<p>
	@param	source_messenger	The source Messenger that is initiating
		the Messenger link. If null nothing is done.
	@param	message	The Message contain the address of the Messenger to
		be linked. If null nothing is done.
	@see	#Messenger_Unlink(Messenger, Message)
*/
protected void Messenger_Link
	(
	Messenger	source_messenger,
	Message		message
	)
{
if (source_messenger == null ||
	message == null)
	return;

if (message.Find (ACK_ACTION) != null ||
	message.Find (NACK_ACTION) != null)
	return;

String
	linked_address = message.Get (ADDRESS_PARAMETER_NAME);
Messenger
	linked_messenger = Lookup_Messenger (linked_address);
if (linked_messenger == null)
	{
	String
		explanation
			= "Can't link to a Messenger with an " + ADDRESS_PARAMETER_NAME
				+ " of " + linked_address;
	Log_Time (explanation + NL
			+"in request from: "
				+ source_messenger.Address () + NL
			+ source_messenger + NL
			+ message);

	//	Add the failure explanation to the message.
	message
		.Set_Token (NACK_ACTION, 0)
		.Set (EXPLANATION_PARAMETER_NAME, explanation);
	}
else
	{
	Log_Time ("Linking:" + NL
		+ source_messenger + NL
		+ source_messenger.Identity () + NL
		+"and" + NL
		+ linked_messenger + NL
		+ linked_messenger.Identity ());

	//	Add messenger forwarding links.
	source_messenger.Add_Forwarding (linked_messenger);
	linked_messenger.Add_Forwarding (source_messenger);

	message.Set_Token (ACK_ACTION, 0);
	Message
		identity = linked_messenger.Identity ();
	if (identity != null)
		{
		//	Add the linked Messenger identity and routing to the message.
		try {message.Add (IDENTITY_ACTION,
				new Message (identity)
					.Set (ROUTE_TO_PARAMETER_NAME,
						Route (source_messenger, linked_messenger), 0));}
		catch (PVL_Exception exception) {/* Shouldn't happen */}
		}
	else
		message.Add
			(Message.Identity ()
				.Set (ADDRESS_PARAMETER_NAME, linked_address));
	}

//	Send the message back as confirmation.
Send (source_messenger, message.Reply_To (message));
}

/**	Unlink two Messengers.
<p>
	The specified Message must contain an {@link #ADDRESS_PARAMETER_NAME}
	value that specifies the address of {@link #Lookup_Messenger(String)
	known Messenger} from which the source Messenger is to be unlinked. The
	linked Messenger is {@link Messenger#Remove_Forwarding(Messenger) removed
	from the forwarding list} of the source Messenger and the source
	Messenger is removed from the forwarding list of the linked Messenger.
<p>
	<b>N.B.</b>: No reply is sent to the source Messenger.
<p>
	@param	source_messenger	The source Messenger that is initiating
		the Messenger link. If null nothing is done.
	@param	message	The Message contain the address of the Messenger to
		be linked. If null nothing is done.
	@see	#Messenger_Link(Messenger, Message)
*/
protected void Messenger_Unlink
	(
	Messenger	source_messenger,
	Message		message
	)
{
if (source_messenger == null ||
	message == null)
	return;

String
	linked_address = message.Get (ADDRESS_PARAMETER_NAME);
Messenger
	linked_messenger = Lookup_Messenger (linked_address);
String
	report =
	"Unlinking:" + NL
	+ source_messenger + NL
	+ source_messenger.Identity () + NL
	+"and" + NL;
if (linked_messenger != null)
	report +=
	  linked_messenger + NL
	+ linked_messenger.Identity ();
else
	report +=
	"Messenger at address " + linked_address + " (not present)";
Log_Time (report);

//	Remove messenger forwarding links.
source_messenger.Remove_Forwarding (linked_messenger);
if (linked_messenger != null)
	linked_messenger.Remove_Forwarding (source_messenger);
}


protected static Value Route
	(
	Messenger	source_messenger,
	Messenger	client_messenger
	)
{
if (source_messenger == null ||
	client_messenger == null)
	return null;
Message
	identity = source_messenger.Identity ();
if (identity == null)
	//	No source identity.
	return null;
Value
	source_addresses = identity.Route_To ();
if (source_addresses.Array_Size () == 0)
	//	No source route-to list.
	return null;
//	Dispatcher-to-source reversed to source-to-Dispatcher.
try {Collections.reverse (source_addresses.Vector_Data ());}
catch (PVL_Exception exception)
	{
	//	Shouldn't happen.
	return null;
	}

identity = client_messenger.Identity ();
if (identity == null)
	//	No client identity.
	return null;
Value
	client_addresses = identity.Route_To ();
if (client_addresses.Array_Size () == 0)
	//	No client route-to list.
	return null;
try {client_addresses.Add (source_addresses.Vector_Data ());}
catch (PVL_Exception exception)
	{
	//	Shouldn't happen.
	return null;
	}
return client_addresses;
}

/**	Get the Message route-to values for a Messenger.
<p>
	The route_to values specify the {@link Message#Route_To(Value)
	Message routing addresses} used by a Messenger when a Message
	is {@link #Send(Messenger, Message) sent}.
<p>
	@param	messenger	A Messenger for which to get the Message route-to
		address values
	@return	An Array Value containing the route-to address String Values.
*/
public static Value Route_To
	(
	Messenger	messenger
	)
{
Message
	identity = messenger.Identity ();
if (identity != null)
	return identity.Route_To ();
return null;
}

/**	Send a status report message.
<p>
	The status report contains the Dispatcher {@link #Identity()
	identity} plus its communications {@link
	#IDENTIFY_TIMEOUT_PARAMETER_NAME}, {@link #PORT_PARAMETER_NAME},
	#HELLO_PORT_PARAMETER_NAME} and {@link #HELLO_ADDRESS_PARAMETER_NAME}
	parameters.
<p>
	The Dispatcher identification is followed by a {@link
	#MESSENGER_STATUS_PARAMETER_NAME} group containing each connected
	Messenger's {@link Messenger#Identity() identity} with the identities
	of each of its {@link Messenger#Forwarding_Messengers() forwarding
	Messengers} and its Message statistics:
<p><ul>
<li>{@link #MESSAGES_SENT_PARAMETER_NAME}
	from {@link Messenger#Messages_Sent()}
<li>{@link #MESSAGE_BYTES_SENT_PARAMETER_NAME}
	from {@link Messenger#Message_Bytes_Sent()}
<li>{@link #MESSAGES_SENT_DROPPED_PARAMETER_NAME}
	from {@link Messenger#Messages_Sent_Dropped()}
<li>{@link #MESSAGES_SENT_CORRUPTED_PARAMETER_NAME}
	from {@link Messenger#Messages_Sent_Corrupted()}
<li>{@link #MESSAGES_RECEIVED_PARAMETER_NAME}
	from {@link Messenger#Messages_Received()}
<li>{@link #MESSAGE_BYTES_RECEIVED_PARAMETER_NAME}
	from {@link Messenger#Message_Bytes_Received()}
<li>{@link #MESSAGES_RECEIVED_DROPPED_PARAMETER_NAME}
	from {@link Messenger#Messages_Received_Dropped()}
<li>{@link #MESSAGES_RECEIVED_CORRUPTED_PARAMETER_NAME}
	from {@link Messenger#Messages_Received_Corrupted()}
<li>{@link #MESSAGES_UNFORWARDABLE_PARAMETER_NAME}
	from {@link Messenger#Messages_Unforwardable()}
</ul><p>
	Each Messenger that registered an {@link Messenger#Error_Condition()
	error condition} will also have an {@link #EXCEPTION_PARAMETER_NAME}
	describing the exception following the Message statistics.
<p>
	Following the Messenger identities a {@link #MEMORY_STATUS_PARAMETER_NAME}
	group contains memory use parameters:
<p><ul>
<li>{@link #MEMORY_AVAILABLE_PARAMETER_NAME} - the maximum amount of
	memory available to the Java Runtime Environment.
<li>{@link #MEMORY_ALLOCATED_PARAMETER_NAME} - the amount of memory
	currently allocated by the application.
<li>{@link #MEMORY_FREE_PARAMETER_NAME} - the amount of allocated memory
	currently unused.
</ul><p>
	@param	source_messenger	The Messagner to use to send the status
		report Message.
	@param	message	The Message to {@link Message#Reply_To(Message) reply
		to}.
*/
protected void Status_Report
	(
	Messenger	source_messenger,
	Message		message
	)
{
Message
	status_report =
		Identity ()
			.Set (ACTION_PARAMETER_NAME, STATUS_REPORT_ACTION)
			.Set (CONFIGURATION_SOURCE_PARAMETER_NAME,
				The_Configuration.Source ())
			.Set (IDENTIFY_TIMEOUT_PARAMETER_NAME, Identify_Timeout)
			.Set (PORT_PARAMETER_NAME, Port)
			.Set (HELLO_PORT_PARAMETER_NAME, Hello_Port)
			.Set (HELLO_ADDRESS_PARAMETER_NAME, Hello_Address)
			.Reply_To (message),
	messenger_status = new Message ();
messenger_status.Name (MESSENGER_STATUS_PARAMETER_NAME);
synchronized (Messengers)
	{
	Messenger
		messenger;
	Message
		identity = null;
	Exception
		error_condition;
	int
		index = -1,
		total = Messengers.size ();
	while (++index < total)
		{
		messenger = Messengers.get (index);
		try {identity = new Message (messenger.Identity ());}
		catch (PVL_Exception exception) {}
		identity.Name (identity.Get (NAME_PARAMETER_NAME));
		identity
			.Add (messenger.Forwarding_Messengers ())
			.Set (MESSAGES_SENT_PARAMETER_NAME,
				messenger.Messages_Sent ())
			.Set (MESSAGE_BYTES_SENT_PARAMETER_NAME,
				messenger.Message_Bytes_Sent ())
			.Set (MESSAGES_SENT_DROPPED_PARAMETER_NAME,
				messenger.Messages_Sent_Dropped ())
			.Set (MESSAGES_SENT_CORRUPTED_PARAMETER_NAME,
				messenger.Messages_Sent_Corrupted ())
			.Set (MESSAGES_RECEIVED_PARAMETER_NAME,
				messenger.Messages_Received ())
			.Set (MESSAGE_BYTES_RECEIVED_PARAMETER_NAME,
				messenger.Message_Bytes_Received ())
			.Set (MESSAGES_RECEIVED_DROPPED_PARAMETER_NAME,
				messenger.Messages_Received_Dropped ())
			.Set (MESSAGES_RECEIVED_CORRUPTED_PARAMETER_NAME,
				messenger.Messages_Received_Corrupted ())
			.Set (MESSAGES_UNFORWARDABLE_PARAMETER_NAME,
				messenger.Messages_Unforwardable ());
		error_condition = messenger.Error_Condition ();
		if (error_condition != null)
			identity.Set (EXCEPTION_PARAMETER_NAME,
				error_condition.toString ());

		messenger_status.Add (identity);
		}
	}
status_report.Add (messenger_status);

status_report.Add (new Message ()
	.Set (MEMORY_AVAILABLE_PARAMETER_NAME,
		JRE.maxMemory ())
	.Set (MEMORY_ALLOCATED_PARAMETER_NAME,
		JRE.totalMemory ())
	.Set (MEMORY_FREE_PARAMETER_NAME,
		JRE.freeMemory ())
	.Name (MEMORY_STATUS_PARAMETER_NAME));

Send (source_messenger, status_report);
}


protected void NACK
	(
	Messenger	messenger,
	Message		message
	)
{
Send (messenger,
	Message.NACK ()
		.Add (ORIGINAL_MESSAGE_PARAMETER_NAME, message)
		.Reply_To (message),
	false);	//	No failure report.
}

/*==============================================================================
	Say_Hello Broadcast
*/
private class Say_Hello 
	extends Thread 
{
public void run () 
{
MulticastSocket
	socket;
DatagramPacket
	packet;
try
	{
	socket = new MulticastSocket ();
	socket.setTimeToLive (0);

	//	Assemble the Datagram packet.
	byte[]
		buffer = HELLO_MESSAGE.getBytes ();
	packet = new DatagramPacket
			(buffer, buffer.length,
			InetAddress.getByName (Hello_Address), Hello_Port);
	}
catch (IOException exception)
	{
	Log ("Unable to construct the multicast socket." + NL
		+ exception);
	return;
	}

try
	{
	int
		count = 0;
	while (count++ < HELLO_COUNT) 
		{
		socket.send (packet);
		sleep (HELLO_INTERVAL * 1000);
		}
	}
catch (IOException exception) 
	{
	Log ("There was a problem sending the Hello message." + NL
		+ exception);
	}
catch (InterruptedException exception) {}

socket.close ();
Hello = null;
}
}

/*==============================================================================
	Application main
*/
public static void main
	(
	String[]	args
	)
{
System.out.println (ID + NL);

String
	configuration_filename = null,
	log_filename = null,
	name = "";
int
	port = 0;

for (int count = 0;
	 count < args.length;
	 count++)
	{
	if (args[count].length () > 1 &&
		args[count].charAt (0) == '-')
		{
		switch (args[count].charAt (1))
			{
			case 'C':	//	Configuration
			case 'c':
				if (++count == args.length ||
					args[count].length () == 0 ||
					args[count].charAt (0) == '-')
					{
					System.out.println ("Missing configuration filename");
					Usage ();
					}
				if (configuration_filename != null)
					{
					System.out.println ("Multiple configuration files -" + NL
						+ configuration_filename + NL
						+ "and" + NL
						+ args[count]);
					Usage ();
					}
				configuration_filename = args[count];
				break;

			case 'L':	//	Logging
			case 'l':
				if (++count == args.length ||
					(args[count].length () > 1 &&
					 args[count].charAt (0) == '-'))
					{
					name = Default_Log_Filename ();
					--count;
					}
				else
					name = args[count];

				if (log_filename != null &&
					! log_filename.equals (name))
					{
					System.out.println ("Multiple log files -" + NL
						+ "    " + log_filename + NL
						+ "  and" + NL
						+ "    " + name);
					Usage ();
					}
				if (name.length () == 0)
					name = STDOUT;
				log_filename = name;
				break;

			case 'P':	//	Port
			case 'p':
				if (++count == args.length ||
					args[count].length () == 0 ||
					args[count].charAt (0) == '-')
					Usage ();
				try {port = Integer.parseInt (args[count]);}
				catch (NumberFormatException exception)
					{
					System.out.println ("Invalid port number: " + args[count]);
					Usage ();
					}
				break;

			default:
				System.out.println
					("Unknown argument: " + args[count]);

			case 'H':	//	Help
			case 'h':
				Usage ();
			}
		}
	else
		{
		System.out.println
			("Unknown argument: " + args[count]);
		Usage ();
		}
	}

Configuration
	configuration = null;
if (configuration_filename != null)
	{
	try {configuration = new Configuration (configuration_filename);}
	catch (IllegalArgumentException exception)
		{
		System.out.println ("Unable to find the configuration file - "
			+ configuration_filename);
		System.exit (EXIT_CONFIG_FILE_PROBLEM);
		}
	catch (Configuration_Exception exception)
		{
		System.out.println ("Could not load the configuration file - "
			+ configuration_filename + NL
			+ exception.getMessage ());
		System.exit (EXIT_CONFIG_FILE_PROBLEM);
		}
	}

Dispatcher
	dispatcher = new Dispatcher (configuration, port);

if (DEBUG != DEBUG_OFF)
	{
	if (log_filename == null)
		log_filename = STDOUT;
	}
if (log_filename != null)
	{
	try {System.out.println (dispatcher.Log_Pathname (log_filename));}
	catch (FileNotFoundException exception)
		{
		System.out.println (exception.getMessage ());
		System.exit (EXIT_NO_LOG_FILE);
		}
	dispatcher.Logging (true);
	}

try {dispatcher.Start ();}
catch (Configuration_Exception exception)
	{
	dispatcher.Log_Time ("Could not load the configuration." + NL
		+ exception.getMessage () + NL);
	System.exit (EXIT_CONFIG_FILE_PROBLEM);
	}

catch (IOException exception)
	{
	dispatcher.Log_Time ("Server I/O failure:" + NL
		+ exception.getMessage () + NL);
	System.exit (EXIT_IO_ERROR);
	}
catch (SecurityException exception)
	{
	dispatcher.Log_Time ("Server security violation:" + NL
		+ exception.getMessage () + NL);
	System.exit (EXIT_SECURITY_VIOLATION);
	}
catch (Exception exception)
	{
	dispatcher.Log_Time ("Server failure:");
	if (dispatcher.Logging ())
		exception.printStackTrace (dispatcher.Log_Stream ());
	dispatcher.Log ();
	System.exit (EXIT_UNKNOWN_EXCEPTION);
	}
}


public static void Usage ()
{
System.out.println 
	("Usage: " + Default_Dispatcher_Name () + " <Options>" + NL
	+"  Options -" + NL
	+"    -Configuration <source>"
		+ " (default: " + Default_Configuration_Source () + ')' + NL
	+"    -Logging [<pathname>]"
		+ " (default: " + Default_Log_Filename () + ')' + NL
	+"    -Port <number>"
		+ " (default: " + DEFAULT_PORT + ')' + NL
	+"    -Help" + NL
	);
System.exit (EXIT_COMMAND_LINE_SYNTAX);
}


}
