/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment drag 'n drop example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>
#include <gst/gst.h>

/* Text regeneration timeout tag */
static guint regenerate_tag = 0;

/* pattern color animation timeout tag */
static guint animation_tag = 0;

/* Blue component of the pattern foreground color */
static guchar blue_component = 20;

/* The GStreamer pipeline */
static GstElement *pipeline = NULL;

/* The pattern image mapping matrix */
static PgmMat4x4 *pattern_matrix;

/* Accepted extension suffixes, not meant to be used in production... */
static const gchar *extensions[] = {
  "png", "jpg", "avi", "mpg", "ogg", "mkv", "mov", "wmv", "flv", "mp3", NULL
};

/* Decrease color handler */
static gboolean
decrease_color_cb (gpointer data)
{
  PgmDrawable *pattern = PGM_DRAWABLE (data);

  if (blue_component == 20)
    {
      animation_tag = 0;
      return FALSE;
    }

  blue_component -= 2;
  pgm_drawable_set_fg_color (pattern, 20, 20, blue_component, 255);
  return TRUE;
}

/* Increase color handler */
static gboolean
increase_color_cb (gpointer data)
{
  PgmDrawable *pattern = PGM_DRAWABLE (data);

  if (blue_component == 120)
    {
      animation_tag = 0;
      return FALSE;
    }

  blue_component += 2;
  pgm_drawable_set_fg_color (pattern, 20, 20, blue_component, 255);
  return TRUE;
}

/* Start color animation */
static void
start_color_animation (GSourceFunc sourcefunc,
                       gpointer data)
{
  if (animation_tag)
    g_source_remove (animation_tag);

  animation_tag = g_timeout_add (10, sourcefunc, data);
}

/* Drag motion handler */
static gboolean
drag_motion_event_cb (PgmViewport *viewport,
                      PgmEventDnd *event,
                      gpointer data)
{
  const gchar **extension = extensions;
  gchar *uri = *event->uri;

  /* Check if the file is a media, based on the extension */
  while (*extension)
    {
      if (g_str_has_suffix (uri, *extension))
        {
          gfloat x, y, z;

          /* Convert 2d coordinates to canvas coordiantes */
          pgm_viewport_to_canvas (viewport, &x, &y, &z,
                                  event->x, event->y, -1.0f);

          /* Only accept the drag if it's on the drop area */
          if (x > 75.0f && x < 565.0f && y > 75.0f && y < 405.0f)
            {
              start_color_animation (increase_color_cb, data);
              return TRUE;
            }
        }
      extension++;
    }

  start_color_animation (decrease_color_cb, data);

  return FALSE;
}

/* Drag drop handler */
static void
drag_drop_event_cb (PgmViewport *viewport,
                    PgmEventDnd *event,
                    gpointer data)
{
  start_color_animation (decrease_color_cb, data);

  /* Set the new pipeline uri, and play the media */
  gst_element_set_state (pipeline, GST_STATE_READY);
  g_object_set (G_OBJECT (pipeline), "uri", *event->uri, NULL);
  gst_element_set_state (pipeline, GST_STATE_PLAYING);
}

/* Drag leave handler */
static void
drag_leave_event_cb (PgmViewport *viewport,
                     PgmEventDnd *event,
                     gpointer data)
{
  start_color_animation (decrease_color_cb, data);
}

/* Text regeneration handler */
static gboolean
regenerate_cb (gpointer data)
{
  PgmDrawable *text = PGM_DRAWABLE (data);

  pgm_drawable_regenerate (text);
  regenerate_tag = 0;

  return FALSE;
}

/* Configure event handler */
static void
configure_event_cb (PgmViewport *viewport,
                    PgmEventConfigure *event,
                    gpointer data)
{
  /* Remove current regeneration timeout */
  if (regenerate_tag)
    g_source_remove (regenerate_tag);

  /* Then add the new regeneration timeout */
  regenerate_tag = g_timeout_add (300, regenerate_cb, data);
}

/* Translate the mapping matrix of the pattern */
static gboolean
update_pass_cb (PgmViewport *viewport,
                gpointer data)
{
  PgmImage *image = PGM_IMAGE (data);

  pgm_mat4x4_translate_from_scalars (pattern_matrix, -0.004f, -0.0025f, 0.0f);
  pgm_image_set_mapping_matrix (PGM_IMAGE (image), pattern_matrix);

  return TRUE;
}

/* Key press event handler */
static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  static gboolean fullscreen = FALSE;

  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Fullscreen */
        case PGM_f:
          fullscreen = !fullscreen;
          pgm_viewport_set_fullscreen (viewport, fullscreen);
          break;

          /* Quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

        default:
          break;
        }
    }
}

/* Entry point */
int
main (int argc,
      char *argv[])
{
  PgmViewport *viewport = NULL;
  PgmDrawable *pattern, *text, *media;
  GstElement *image_sink, *visualisation;
  PgmCanvas *canvas;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_title (viewport, "Drag 'n drop");
  pgm_viewport_set_size (viewport, 640, 480);

  /* Pattern creation */
  pattern = pgm_image_new_from_file ("examples/pictures/square-pattern.png", 0);
  pgm_drawable_set_size (pattern, 490.0f, 330.0f);
  pgm_drawable_set_position (pattern, 75.0f, 75.0f, 0.0f);
  pgm_drawable_set_bg_color (pattern, 255, 255, 255, 0);
  pgm_drawable_set_fg_color (pattern, 20, 20, blue_component, 255);
  pgm_image_set_layout (PGM_IMAGE (pattern), PGM_IMAGE_FILLED);
  pgm_image_set_wrapping (PGM_IMAGE (pattern), PGM_IMAGE_REPEAT,
                          PGM_IMAGE_REPEAT);
  pattern_matrix = pgm_mat4x4_new_identity ();
  pgm_mat4x4_scale_from_scalars (pattern_matrix, 4900.0f / 330.0f, 10.0f, 0.0f);
  pgm_image_set_mapping_matrix (PGM_IMAGE (pattern), pattern_matrix);
  pgm_drawable_show (pattern);

  /* Text creation */
  text = pgm_text_new ("<span face='MgOpen Modata' weight='bold'>"
                       "Drag 'n drop</span>");
  pgm_text_set_weight (PGM_TEXT (text), PGM_TEXT_WEIGHT_BOLD);
  pgm_text_set_alignment (PGM_TEXT (text), PGM_TEXT_ALIGN_CENTER);
  pgm_text_set_font_height (PGM_TEXT (text), 70.0f);
  pgm_drawable_set_size (text, 490.0f, 80.0f);
  pgm_drawable_set_position (text, 75.0f, 200.0f, 0.0f);
  pgm_drawable_set_bg_color (text, 255, 255, 255, 0);
  pgm_drawable_set_fg_color (text, 255, 255, 255, 175);
  pgm_drawable_show (text);

  /* Media creation */
  media = pgm_image_new ();
  pgm_drawable_set_size (media, 490.0f, 330.0f);
  pgm_drawable_set_position (media, 75.0f, 75.0f, 0.0f);
  pgm_drawable_set_bg_color (media, 255, 255, 255, 0);
  pgm_drawable_set_fg_color (media, 255, 255, 255, 175);
  pgm_drawable_show (media);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 640.0f, 480.0f);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add_many (canvas, PGM_DRAWABLE_MIDDLE, pattern, text, media, NULL);

  /* GStreamer pipeline setup */
  pipeline = gst_element_factory_make ("playbin", NULL);
  image_sink = gst_element_factory_make ("pgmimagesink", NULL);
  visualisation = gst_element_factory_make ("goom", NULL);
  g_object_set (G_OBJECT (pipeline), "vis-plugin", visualisation, NULL);
  g_object_set (G_OBJECT (pipeline), "video-sink", image_sink, NULL);
  g_object_set (G_OBJECT (image_sink), "image", media, NULL);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  g_signal_connect (G_OBJECT (viewport), "configure-event",
                    G_CALLBACK (configure_event_cb), text);
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "drag-motion-event",
                    G_CALLBACK (drag_motion_event_cb), pattern);
  g_signal_connect (G_OBJECT (viewport), "drag-drop-event",
                    G_CALLBACK (drag_drop_event_cb), pattern);
  g_signal_connect (G_OBJECT (viewport), "drag-leave-event",
                    G_CALLBACK (drag_leave_event_cb), pattern);
  g_signal_connect (G_OBJECT (viewport), "update-pass",
                    G_CALLBACK (update_pass_cb), pattern);
  pgm_viewport_show (viewport);
  pgm_main ();

  /* Deinit */
  pgm_mat4x4_free (pattern_matrix);
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
