/* $Id: audio.c,v 1.9 2003/06/13 09:46:10 sjoerd Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>

#include <SDL.h>
#include <SDL_mixer.h>

#include "audio.h"
#include "global.h"
#include "list.h"

char *audio_effectnames[] = {"throw",
                             "jump",
                             "deliver",
                             "pickup",
                             "hit",
                             "powerup",
                             "dead"
};

Mix_Chunk *
audio_loadwav(int effectnr) {
  char file[MAXPATHLEN];
  Mix_Chunk *result;
  snprintf(file,MAXPATHLEN,"%s/ffrenzy/sounds/ffrenzy_%s_sound.wav",
           DATADIR,
           audio_effectnames[effectnr]);

  result = Mix_LoadWAV(file);
  if (result == NULL) { 
    WARN("Couldn't load %s sound from %s",audio_effectnames[effectnr],file);
  }
  return result;
}

Mix_Music *
audio_loadmusic(int num) {
  char dirname[MAXPATHLEN];
  char file[MAXPATHLEN];
  struct stat dirstat;
  struct dirent *direntry;
  DIR *dir;
  int entries;

  snprintf(dirname,MAXPATHLEN,"%s/ffrenzy/music",
           DATADIR);
  if (stat(dirname,&dirstat) == -1) {
    WARN("Couldn't stat music dir: %s",strerror(errno));
    return NULL;
  }
  if (! S_ISDIR(dirstat.st_mode)) {
    WARN("Music dir isn't a dir");
    return NULL;
  }
  dir = opendir(dirname); 
  if (dir == NULL) {
    WARN("Couldn't open music dir: %s",strerror(errno));
    return NULL;
  }
  for (entries = 0; readdir(dir) != NULL; entries++) 
    ;
  if (entries <= 2) {
    /* only . and .. in the music directory */
    return NULL;
  }
  num %= (entries - 2);
  seekdir(dir,0);
  direntry = NULL;
  do {
    direntry = readdir(dir);
    if (direntry == NULL) {
      /* size of the dir apperently changed
       * try loading again
       */
      return audio_loadmusic(num);
    }
    if (strcmp(direntry->d_name,".") && strcmp(direntry->d_name,".."))  {
      num--;
    }
  } while (num >= 0);

  if (direntry == NULL)  {
    return NULL;
  }

  snprintf(file,MAXPATHLEN,"%s/ffrenzy/music/%s",
           DATADIR,direntry->d_name);
  return Mix_LoadMUS(file);
}


Audio *
new_audio(int play_music) {
  int audio_rate = 44100;
  Uint16 audio_format = AUDIO_S16;
  int audio_channels = 2;
  int audio_buffers = 2048;
  int x;
  Audio *result ;

  assert(EV_NREFFECTS == (sizeof(audio_effectnames)/sizeof(char *)));
  if (Mix_OpenAudio(audio_rate, audio_format, audio_channels, audio_buffers)) {
     WARN("Unable to open audio!");
     return NULL;
  }
  Mix_AllocateChannels(8);
  result = malloc(sizeof(Audio));
  assert(result!=NULL);
  for (x=EV_FIRST ; x < EV_NREFFECTS; x++) {
    result->sounds[x] = audio_loadwav(x);
  }
  if (play_music) {
    result->music_track = 0;
    result->music = audio_loadmusic(0);
    if (result->music != NULL) {
      Mix_PlayMusic(result->music,1);
    } else {
      WARN(_("No music tracks found"));
    }
  } else {
    result->music = NULL;
  }
  return result;
}

void
del_audio(Audio *audio) {
  int x;

  for (x=EV_FIRST; x < EV_NREFFECTS; x++) {
    if (audio->sounds[x] == NULL) {
      Mix_FreeChunk(audio->sounds[x]);
    }
  }
  if (audio->music != NULL) { 
    Mix_FreeMusic(audio->music);
  }
  free(audio);
}

int
do_audio_update(void *data, void *user_data) {
  Model_object *obj = (Model_object *) data;
  Audio *a = (Audio *) user_data;
  int x;
  if (obj->events) {
    for (x=EV_FIRST; x < EV_NREFFECTS ;x++) {
      if ((obj->events) & (1 << x)) {
        Mix_PlayChannel(-1,a->sounds[x],0);
      }
    }
  }
  return TRUE;
}

void
audio_update(Audio *a,Model *m) {
  model_foreach_object(m,do_audio_update,a);
  if (a->music != NULL && !Mix_PlayingMusic()) {
    a->music_track++;
    Mix_FreeMusic(a->music);
    a->music = audio_loadmusic(a->music_track);
    if (a->music != NULL) {
      Mix_PlayMusic(a->music,1);
    }
  }
}

