/*
 * $Id: isa_gen_ne2000_glue.c,v 1.14 2012-02-22 09:27:20 siflkres Exp $
 *
 * Copyright (C) 2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue.h"

#include "isa_gen_ne2000_glue.h"

#define CHIP_(x) isa_gen_ne2000_glue_ ## x

struct cpssp {
	/*
	 * Config
	 */
	unsigned int irq;
	unsigned int ioaddr;

	/*
	 * Ports
	 */
	struct sig_boolean_or *port_pc_irq7;
	struct sig_boolean_or *port_pc_irq9;
	struct sig_boolean_or *port_pc_irq10;
	struct sig_boolean_or *port_pc_irq11;
	struct sig_boolean_or *port_pc_irq12;
	struct sig_boolean_or *port_pc_irq14;
	struct sig_boolean *port_nic_reset_hash_;
	unsigned int state_power;
	unsigned int state_reset_hash_;
	struct sig_isa_bus_main *port_nic_bus;
	struct sig_isa_bus_dma *port_nic_dma;
	struct sig_cs *port_ram;
	struct sig_cs *port_rom;
};

static int
CHIP_(inb)(void *_cpssp, uint8_t *valp, uint16_t port)
{
	struct cpssp *cpssp = _cpssp;

	if (cpssp->ioaddr <= port
	 && port < cpssp->ioaddr + 0x10) {
		return sig_isa_bus_inb(cpssp->port_nic_bus, cpssp,
				valp, port & 0xf);

	} else if (port == cpssp->ioaddr + 0x10) {
		return sig_isa_bus_dma_ack_inb(cpssp->port_nic_dma, cpssp,
				0, valp);

	} else {
		return -1;
	}
}

static int
CHIP_(outb)(void *_cpssp, uint8_t val, uint16_t port)
{
	struct cpssp *cpssp = _cpssp;

	if (cpssp->ioaddr <= port
	 && port < cpssp->ioaddr + 0x10) {
		return sig_isa_bus_outb(cpssp->port_nic_bus, cpssp,
				val, port & 0xf);

	} else if (port == cpssp->ioaddr + 0x10) {
		return sig_isa_bus_dma_ack_outb(cpssp->port_nic_dma, cpssp,
				0, val);

	} else if (port == cpssp->ioaddr + 0x1f) {
		sig_boolean_set(cpssp->port_nic_reset_hash_, cpssp, 0);
		sig_boolean_set(cpssp->port_nic_reset_hash_, cpssp, 1);
		return 0;

	} else {
		return -1;
	}
}

static int
CHIP_(inw)(void *_cpssp, uint16_t *valp, uint16_t port)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t val0;
	uint8_t val1;

	if (port == cpssp->ioaddr + 0x10) {
		sig_isa_bus_dma_ack_inb(cpssp->port_nic_dma, cpssp, 0, &val0);
		sig_isa_bus_dma_ack_inb(cpssp->port_nic_dma, cpssp, 0, &val1);
		*valp = (val1 << 8) | (val0 << 0);
		return 0;
	} else {
		return -1;
	}
}

static int
CHIP_(outw)(void *_cpssp, uint16_t val, uint16_t port)
{
	struct cpssp *cpssp = _cpssp;

	if (port == cpssp->ioaddr + 0x10) {
		sig_isa_bus_dma_ack_outb(cpssp->port_nic_dma, cpssp,
				0, (val >> 0) & 0xff);
		sig_isa_bus_dma_ack_outb(cpssp->port_nic_dma, cpssp,
				0, (val >> 8) & 0xff);
		return 0;
	} else {
		return -1;
	}
}

static void
CHIP_(nic_irq_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	switch (cpssp->irq) {
	case 7: sig_boolean_or_set(cpssp->port_pc_irq7, cpssp, val); break;
	case 9: sig_boolean_or_set(cpssp->port_pc_irq9, cpssp, val); break;
	case 10: sig_boolean_or_set(cpssp->port_pc_irq10, cpssp, val); break;
	case 11: sig_boolean_or_set(cpssp->port_pc_irq11, cpssp, val); break;
	case 12: sig_boolean_or_set(cpssp->port_pc_irq12, cpssp, val); break;
	case 14: sig_boolean_or_set(cpssp->port_pc_irq14, cpssp, val); break;
	default: assert(0);
	}
}

static int
CHIP_(readb)(void *_cpssp, uint32_t addr, uint8_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if ((addr >> 14) & 1) {
		sig_cs_readb(cpssp->port_ram, cpssp, valp, addr);
	} else {
		sig_cs_readb(cpssp->port_rom, cpssp, valp, addr);
	}

	return 0;
}

static int
CHIP_(writeb)(void *_cpssp, uint32_t addr, uint8_t val)
{
	struct cpssp *cpssp = _cpssp;

	if ((addr >> 14) & 1) {
		sig_cs_writeb(cpssp->port_ram, cpssp, val, addr);
	}

	return 0;
}

static void
CHIP_(power_set)(void *_cpssp, unsigned int val)
{
	/* FIXME */
}

static void
CHIP_(reset_set)(void *_cpssp, unsigned int val)
{
	/* FIXME */
}

void *
CHIP_(create)(
	const char *name,
	const char *irq,
	const char *ioaddr,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_reset_hash_,
	struct sig_isa_bus_main *port_pc_bus,
	struct sig_boolean_or *port_pc_irq7,
	struct sig_boolean_or *port_pc_irq9,
	struct sig_boolean_or *port_pc_irq10,
	struct sig_boolean_or *port_pc_irq11,
	struct sig_boolean_or *port_pc_irq12,
	struct sig_boolean_or *port_pc_irq14,
	struct sig_isa_bus_main *port_nic_bus,
	struct sig_isa_bus_dma *port_nic_dma,
	struct sig_boolean_or *port_nic_irq,
	struct sig_boolean *port_nic_reset_hash_,
	struct sig_cs *port_ram,
	struct sig_cs *port_rom
)
{
	static unsigned int irq_default = 9;
	static unsigned int ioaddr_default = 0x300;
	static const struct sig_boolean_funcs power_funcs = {
		.set = CHIP_(power_set),
	};
	static const struct sig_boolean_funcs reset_hash__funcs = {
		.set = CHIP_(reset_set),
	};
	static const struct sig_isa_bus_main_funcs pc_bus_funcs = {
		.inb = CHIP_(inb),
		.inw = CHIP_(inw),
		.outb = CHIP_(outb),
		.outw = CHIP_(outw),
	};
	static const struct sig_isa_bus_main_funcs nic_bus_funcs = {
		.readb = CHIP_(readb),
		.writeb = CHIP_(writeb),
	};
	static const struct sig_isa_bus_dma_funcs nic_dma_funcs = {
		/* FIXME */
	};
	static const struct sig_boolean_or_funcs nic_irq_funcs = {
		.set = CHIP_(nic_irq_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Get Config */
	if (irq == NULL
	 || atoi(irq) == -1) {
		cpssp->irq = irq_default;
		switch (irq_default) {
		case 9: irq_default = 10; break;
		case 10: irq_default = 11; break;
		case 11: irq_default = 12; break;
		case 12: irq_default = 14; break;
		case 14: irq_default = 7; break;
		case 7: irq_default = 9; break;
		default: assert(0);
		}
	} else {
		cpssp->irq = strtoul(irq, NULL, 0);
	}
	if (ioaddr == NULL
	 || atoi(ioaddr) == -1) {
		cpssp->ioaddr = ioaddr_default;
		switch (ioaddr_default) {
		case 0x300: ioaddr_default = 0x280; break;
		case 0x280: ioaddr_default = 0x320; break;
		case 0x320: ioaddr_default = 0x380; break;
		case 0x380: ioaddr_default = 0x360; break;
		case 0x360: ioaddr_default = 0x340; break;
		case 0x340: ioaddr_default = 0x300; break;
		default: assert(0);
		}
	} else {
		cpssp->ioaddr = strtoul(ioaddr, NULL, 0);
	}

	/* FIXME */

	/* Call */
	sig_isa_bus_main_connect(port_pc_bus, cpssp, &pc_bus_funcs);

	cpssp->port_nic_bus = port_nic_bus;
	sig_isa_bus_main_connect(port_nic_bus, cpssp, &nic_bus_funcs);

	cpssp->port_nic_dma = port_nic_dma;
	sig_isa_bus_dma_connect(port_nic_dma, cpssp, &nic_dma_funcs);

	cpssp->port_ram = port_ram;
	cpssp->port_rom = port_rom;

	/* Out */
	cpssp->port_pc_irq7 = port_pc_irq7;
	sig_boolean_or_connect_out(port_pc_irq7, cpssp, 0);

	cpssp->port_pc_irq9 = port_pc_irq9;
	sig_boolean_or_connect_out(port_pc_irq9, cpssp, 0);

	cpssp->port_pc_irq10 = port_pc_irq10;
	sig_boolean_or_connect_out(port_pc_irq10, cpssp, 0);

	cpssp->port_pc_irq11 = port_pc_irq11;
	sig_boolean_or_connect_out(port_pc_irq11, cpssp, 0);

	cpssp->port_pc_irq12 = port_pc_irq12;
	sig_boolean_or_connect_out(port_pc_irq12, cpssp, 0);

	cpssp->port_pc_irq14 = port_pc_irq14;
	sig_boolean_or_connect_out(port_pc_irq14, cpssp, 0);

	cpssp->port_nic_reset_hash_ = port_nic_reset_hash_;
	sig_boolean_connect_out(port_nic_reset_hash_, cpssp, 0);

	/* In */
	cpssp->state_power = 0;
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);

	cpssp->state_reset_hash_ = 0;
	sig_boolean_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	sig_boolean_or_connect_in(port_nic_irq, cpssp, &nic_irq_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
