/* $Id: glue-io.c,v 1.8 2009-05-13 09:45:40 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <sys/types.h>
#include <sys/time.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
/* includes for cim_sched_yield() */
#if defined(LINUX)
#include <sched.h>
#endif

#include "glue-io.h"

static struct {
	int fd;
	void *s;
	void (*f)(int, void *);
} callback[100];
static unsigned int callback_count;


static void
io_sched_yield(void)
{
#if defined(LINUX)
	sched_yield();
#elif defined(OPENBSD) || defined(DARWIN)
	(void) getpid();	/* FIXME */
#else
#error "Unsupported OS."
#endif
}

int
io_write(int fd, const void *_buf, int buflen)
{
	const char *buf = (const char *) _buf;
	int count;
	int ret;

	for (count = 0; count < buflen; count += ret) {
		ret = write(fd, buf + count, buflen - count);

		if (ret <= 0) {
			if (ret == -1
			 && (errno == EAGAIN
			  || errno == EINTR)) {
				io_sched_yield();
				ret = 0;
			} else {
				return ret;
			}
		}
	}

	return count;
}

int
io_read(int fd, void *_buf, int buflen)
{
	char *buf = (char *) _buf;
	int count;
	int ret;

	for (count = 0; count < buflen; count += ret) {
		ret = read(fd, buf + count, buflen - count);

		if (ret <= 0) {
			/* only retry if we already read a packet fraction */
			if ((ret == 0
			  || errno == EAGAIN
			  || errno == EINTR)
			 && count > 0) {
				io_sched_yield();
				ret = 0;
			} else {
				return ret;
			}
		}
	}

	return count;
}

void
io_register(int fd, void *s, void (*f)(int, void *))
{
	int flags;
	int ret;

	assert(0 <= fd);
	assert(s);
	assert(f);

	assert(callback_count < sizeof(callback) / sizeof(callback[0]));

	callback[callback_count].fd = fd;
	callback[callback_count].s = s;
	callback[callback_count].f = f;
	callback_count++;

	flags = fcntl(fd, F_GETFL, 0);
	assert(0 <= flags);

	flags |= O_NONBLOCK | O_ASYNC;

	ret = fcntl(fd, F_SETFL, flags);
	assert(0 <= ret);

	ret = fcntl(fd, F_SETOWN, getpid());
	assert(0 <= ret);
}

void
io_unregister(int fd)
{
	unsigned int i;

	for (i = 0; ; i++) {
		assert(i < callback_count);
		if (callback[i].fd == fd) {
			break;
		}
	}
	assert(/* 0 <= i && */ i < callback_count);

	callback[i].fd = callback[callback_count - 1].fd;
	callback[i].s = callback[callback_count - 1].s;
	callback[i].f = callback[callback_count - 1].f;

	callback_count--;
}

void
io_do(void)
{
	fd_set rfds;
	struct timeval tv;
	int max;
	unsigned int i;
	int ret;

	do {
		FD_ZERO(&rfds);
		max = -1;
		for (i = 0; i < callback_count; i++) {
			FD_SET(callback[i].fd, &rfds);
			if (max < callback[i].fd) {
				max = callback[i].fd;
			}
		}
		tv.tv_sec = 0;
		tv.tv_usec = 0;

		ret = select(max + 1,
			&rfds, (fd_set *) 0, (fd_set *) 0,
			&tv);
	} while (ret < 0 && errno == EINTR);
	assert(0 <= ret);

	for (i = 0; i < callback_count; i++) {
		if (FD_ISSET(callback[i].fd, &rfds)) {
			(*callback[i].f)(callback[i].fd, callback[i].s);
		}
	}
}

void
io_init(void)
{
	callback_count = 0;
}

void
io_exit(void)
{
}
