#!/usr/bin/env python
# Copyright (c) 2014 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

import cStringIO
import fnmatch
import optparse
import os
import re
import sys

VALID_CHANNELS = ('stable', 'beta', 'dev')

ROOT_FILE_CONTENTS = """.. _pepper_%(channel)s_index:

:orphan:

.. DO NOT EDIT! This document is auto-generated by doxygen/rst_index.py.

########################################
Pepper API Reference (%(channel_title)s)
########################################

This page lists the API for Pepper %(version)s. Apps that use this API can
run in Chrome %(version)s or higher.

:ref:`Pepper C API Reference <pepper_%(channel)s_c_index>`
===========================================================

:ref:`Pepper C++ API Reference <pepper_%(channel)s_cpp_index>`
===============================================================

"""

C_FILE_CONTENTS = """.. _pepper_%(channel)s_c_index:

.. DO NOT EDIT! This document is auto-generated by doxygen/rst_index.py.

##########################################
Pepper C API Reference (%(channel_title)s)
##########################################

This page lists the C API for Pepper %(version)s. Apps that use this API can
run in Chrome %(version)s or higher.

`Interfaces <group___interfaces.html>`_
=======================================
%(interfaces)s

`Structures <group___structs.html>`_
====================================
%(structures)s

`Functions <group___functions.html>`_
=====================================

`Enums <group___enums.html>`_
=============================

`Typedefs <group___typedefs.html>`_
===================================

`Macros <globals_defs.html>`_
=============================

Files
=====
%(files)s
"""

C_INTERFACE_WILDCARDS =  ['struct_p_p_p__*', 'struct_p_p_b__*']

C_STRUCT_WILDCARDS =  ['struct_p_p__*', 'union_p_p__*']

CPP_FILE_CONTENTS = """.. _pepper_%(channel)s_cpp_index:

.. DO NOT EDIT! This document is auto-generated by doxygen/rst_index.py.

############################################
Pepper C++ API Reference (%(channel_title)s)
############################################

This page lists the C++ API for Pepper %(version)s. Apps that use this API can
run in Chrome %(version)s or higher.

`Classes <inherits.html>`_
==========================
%(classes)s

Files
=====
%(files)s
"""

CPP_CLASSES_WILDCARDS = ['classpp_1_1*.html']
CPP_CLASSES_EXCLUDES = ['*-members*']

FILE_WILDCARDS = ['*_8h.html']


def GetName(filename):
  filename = os.path.splitext(filename)[0]
  out = ''
  if filename.startswith('struct_p_p_b__'):
    mangle = filename[7:]  # skip "struct_"
  elif filename.startswith('struct_p_p_p__'):
    mangle = filename[7:]  # skip "struct_"
  elif filename.startswith('struct_p_p__'):
    mangle = filename[7:]  # skip "struct_"
  elif filename.startswith('union_p_p__'):
    mangle = filename[6:]  # skip "union_"
  elif filename.startswith('classpp_1_1_'):
    mangle = filename[12:]
  elif filename.startswith('classpp_1_1ext_1_1_'):
    out = 'Ext::'      # maybe 'ext::' ?
    mangle = filename[19:]
  elif filename.startswith('classpp_1_1internal_1_1_'):
    out = 'Internal::' # maybe 'internal::'
    mangle = filename[24:]
  elif filename.startswith('structpp_1_1internal_1_1_'):
    out = 'Internal::'
    mangle = filename[25:]
  elif filename.endswith('_8h'):
    return filename[:-3].replace('__', '_') + '.h'
  else:
    print 'No match: ' + filename
  cap = True
  for c in mangle:
    if c == '_':
      if cap:
        # If cap is True, we've already read one underscore. The second means
        # that we should insert a literal underscore.
        cap = False
      else:
        cap = True
        continue
    if cap:
      c = c.upper()
      cap = False
    out += c

  # Strip trailing version number (e.g. PPB_Audio_1_1 -> PPB_Audio)
  return re.sub(r'_\d_\d$', '', out)


def GetPath(filepath):
  if os.path.exists(filepath):
    return filepath
  raise OSError('Couldnt find: ' + filepath)


def MakeReSTListFromFiles(path, matches, excludes=None):
  dir_files = os.listdir(path)
  good_files = []
  for match in matches:
    good_files.extend(fnmatch.filter(dir_files, match))

  if excludes:
    for exclude in excludes:
      good_files = [filename for filename in good_files
                    if not fnmatch.fnmatch(filename, exclude)]

  good_files.sort()
  return '\n'.join('  * `%s <%s>`_\n' % (GetName(f), f) for f in good_files)


def MakeTitleCase(s):
  return s[0].upper() + s[1:]


def GenerateRootIndex(channel, version, out_filename):
  channel_title = MakeTitleCase(channel)

  # Use StringIO so we don't write out a partial file on error.
  output = cStringIO.StringIO()
  output.write(ROOT_FILE_CONTENTS % vars())

  with open(out_filename, 'w') as f:
    f.write(output.getvalue())


def GenerateCIndex(root_dir, channel, version, out_filename):
  interfaces = MakeReSTListFromFiles(root_dir, C_INTERFACE_WILDCARDS)
  structures = MakeReSTListFromFiles(root_dir, C_STRUCT_WILDCARDS)
  files = MakeReSTListFromFiles(root_dir, FILE_WILDCARDS)
  channel_title = MakeTitleCase(channel)

  # Use StringIO so we don't write out a partial file on error.
  output = cStringIO.StringIO()
  output.write(C_FILE_CONTENTS % vars())

  with open(out_filename, 'w') as f:
    f.write(output.getvalue())


def GenerateCppIndex(root_dir, channel, version, out_filename):
  classes = MakeReSTListFromFiles(root_dir, CPP_CLASSES_WILDCARDS,
                                  CPP_CLASSES_EXCLUDES)
  files = MakeReSTListFromFiles(root_dir, FILE_WILDCARDS)
  channel_title = MakeTitleCase(channel)

  # Use StringIO so we don't write out a partial file on error.
  output = cStringIO.StringIO()
  output.write(CPP_FILE_CONTENTS % vars())

  with open(out_filename, 'w') as f:
    f.write(output.getvalue())


def main(argv):
  usage = 'Usage: %prog [options] <--root|--c|--cpp> directory'
  parser = optparse.OptionParser(usage=usage)
  parser.add_option('--channel', help='pepper channel (stable, beta, dev)')
  parser.add_option('--version', help='pepper version (e.g. 32, 33, 34, etc.)')
  parser.add_option('--root', help='Generate root API index',
                    action='store_true', default=False)
  parser.add_option('--c', help='Generate C API index', action='store_true',
                    default=False)
  parser.add_option('--cpp', help='Generate C++ API index', action='store_true',
                    default=False)
  parser.add_option('-o', '--output', help='output file.')
  options, files = parser.parse_args(argv)

  if len(files) != 1:
    parser.error('Expected one directory')

  if not options.output:
    parser.error('Need output file')

  if options.channel not in VALID_CHANNELS:
    parser.error('Expected channel to be one of %s' % ', '.join(VALID_CHANNELS))

  if sum((options.c, options.cpp, options.root)) != 1:
    parser.error('Exactly one of --c/--cpp/--root flags is required.')

  root_dir = files[0]

  if options.c:
    GenerateCIndex(root_dir, options.channel, options.version, options.output)
  elif options.cpp:
    GenerateCppIndex(root_dir, options.channel, options.version, options.output)
  elif options.root:
    GenerateRootIndex(options.channel, options.version, options.output)
  else:
    assert(False)
  return 0


if __name__ == '__main__':
  try:
    rtn = main(sys.argv[1:])
  except KeyboardInterrupt:
    sys.stderr.write('%s: interrupted\n' % os.path.basename(__file__))
    rtn = 1
  sys.exit(rtn)
