/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
   \file cdw_dropdown.c

   Implementation of dropdown widget - expandable list of
   items, each item comprised of label and id.
*/


#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "cdw_dropdown.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"


static int      cdw_dropdown_expanded_driver(CDW_DROPDOWN *dropdown);
static cdw_rv_t cdw_dropdown_expand(CDW_DROPDOWN *dropdown);
static cdw_rv_t cdw_dropdown_collapse(CDW_DROPDOWN *dropdown);
static void     cdw_dropdown_display_current_item_reverse(CDW_DROPDOWN *dropdown, bool reverse);




/**
   \brief Create basis for new dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Create basic data structure representing a dropdown, that
   can be filled with dropdown items. You have to call
   cdw_dropdown_add_item() to add items to the dropdown.

   The dropdown is placed in given \p parent window, with geometry
   specified by \p begin_y, \p begin_x and \p width.
   You have to specify maximal number of items in advance, using
   \p n_items_max argument.

   Don't forget to call cdw_dropdown_finalize() after adding all items.

   \param parent - parent window
   \param begin_y - y coordinate of a dropdown in parent window
   \param begin_x - x coordinate of a dropdown in parent window
   \param width - width of a dropdown
   \param n_items_max - expected maximal number of items in the dropdown
   \param colors - color scheme for the dropdown

   \return pointer to new dropdown base structure on success
   \return NULL on failure
*/
CDW_DROPDOWN *cdw_dropdown_new(WINDOW *parent, int begin_y, int begin_x, int width, int n_items_max, cdw_colors_t colors)
{
	cdw_assert (parent, "ERROR: parent window is NULL\n");
	CDW_DROPDOWN *dropdown = (CDW_DROPDOWN *) malloc(sizeof(CDW_DROPDOWN));
	if (!dropdown) {
		cdw_vdm ("ERROR: failed to allocate memory for dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	dropdown->n_items_max = n_items_max;
	dropdown->n_items = 0;

	dropdown->current_item_ind = 0; /* There will be at least one item in the dropdown, so index=0 should be safe. */

	dropdown->widget_id = CDW_WIDGET_ID_DROPDOWN;

	dropdown->items = (cdw_id_label_t **) malloc((size_t) (dropdown->n_items_max + 1) * sizeof (cdw_id_label_t *));

	if (!dropdown->items) {
		cdw_vdm ("ERROR: failed to allocate memory for dropdown items\n");
		free(dropdown);
		dropdown = (CDW_DROPDOWN *) NULL;
		return (CDW_DROPDOWN *) NULL;
	} else {
		for (int i = 0; i < dropdown->n_items_max; i++) {
			dropdown->items[i] = (cdw_id_label_t *) NULL;
		}
	}

	dropdown->parent = parent;
	dropdown->begin_y = begin_y;
	dropdown->begin_x = begin_x;
	dropdown->width = width;
	dropdown->colors = colors;

	dropdown->on_select_callback = (cdw_form_widget_function_t) NULL;

	dropdown->visible = true;

	return dropdown;
}





/**
   \brief Add new item to a dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   1. allocate space for new item and attach it to a dropdown, and
   2. copy label and id into new item, and
   3. increase counter of items in dropdown

   \param dropdown - dropdown to which you want to add an item
   \param id - "id" field of a new item
   \param label - "label" field of a new item

   \return CDW_OK on success
   \return CDW_ERROR on error
*/
cdw_rv_t cdw_dropdown_add_item(CDW_DROPDOWN *dropdown, cdw_id_t id, const char *label)
{
	cdw_assert (dropdown, "ERROR: passing null dropdown to function\n");
	cdw_assert (dropdown->items, "ERROR: dropdown items is null\n");
	cdw_assert (dropdown->n_items < dropdown->n_items_max, "ERROR: trying to create item #%d while maximum is #%d\n",
		    dropdown->n_items, dropdown->n_items_max);

	dropdown->items[dropdown->n_items] = (cdw_id_label_t *) malloc(sizeof (cdw_id_label_t));
	if (!dropdown->items[dropdown->n_items]) {
		cdw_vdm ("ERROR: failed to malloc memory for item #%d\n", dropdown->n_items);
		return CDW_ERROR;
	}

	dropdown->items[dropdown->n_items]->label = strndup(label, (size_t) dropdown->width);
	if (!dropdown->items[dropdown->n_items]->label) {
		cdw_vdm ("ERROR: failed to strndup label #%d: \"%s\"\n", dropdown->n_items, label);
		free(dropdown->items[dropdown->n_items]);
		dropdown->items[dropdown->n_items] = (cdw_id_label_t *) NULL;
		return CDW_ERROR;
	}
	dropdown->items[dropdown->n_items]->id = id;
	dropdown->n_items++;

	return CDW_OK;
}





/**
   \brief Do final calls to few functions to finalize creating a dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Call few ncurses functions to set/initialize some fields of a
   \p dropdown. This prepares the \p dropdown to be usable. This
   should be the last function called during creation of a dropdown.

   \param dropdown - dropdown to be finalized

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_dropdown_finalize(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: passing null dropdown to function\n");

	/* standard code creating ITEM elements */
	dropdown->menu_items = (ITEM **) calloc((size_t) dropdown->n_items + 1, sizeof(ITEM *));
	if (!dropdown->menu_items) {
		cdw_vdm ("ERROR: calloc()\n");
		return CDW_ERROR;
	}

	for (int i = 0; i < dropdown->n_items; i++) {
		dropdown->menu_items[i] = new_item(dropdown->items[i]->label, "");
		if (!dropdown->menu_items[i]) {
			int e = errno;
			cdw_vdm ("ERROR: failed to create menu item for item #%d with label \"%s\", errno = \"%s\"\n",
				 i, dropdown->items[i]->label, strerror(e));
			return CDW_ERROR;
		}
	}
	/* obligatory ending element */
	dropdown->menu_items[dropdown->n_items] = (ITEM *) NULL;

	/* standard code creating MENU element */
	dropdown->menu = new_menu(dropdown->menu_items);
	if (!dropdown->menu) {
		int e = errno;
		cdw_vdm ("ERROR: failed to create menu, errno = \"%s\"\n", strerror(e));
		return CDW_ERROR;
	}

	int parent_x = getbegx(dropdown->parent);
	int parent_y = getbegy(dropdown->parent);
	/* menu should be displayed between opening and closing bracket,
	   so it should be narrower than 'width' and moved a bit right,
	   hence -2 */
	dropdown->menu_window = newwin(dropdown->n_items, dropdown->width - 2,
				       parent_y + dropdown->begin_y,
				       parent_x + dropdown->begin_x + 1);
	if (!dropdown->menu_window) {
		int e = errno;
		cdw_vdm ("ERROR: failed to create dropdown window, errno = \"%s\"\n", cdw_ncurses_error_string(e));
		return CDW_ERROR;
	}
	int rv = set_menu_win(dropdown->menu, dropdown->menu_window);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to set menu window, errno = \"%s\"\n", cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}
	dropdown->menu_window_sub = derwin(dropdown->menu_window, dropdown->n_items, dropdown->width - 2, 0, 0);
	if (!dropdown->menu_window_sub) {
		cdw_vdm ("ERROR: failed to create menu subwindow\n");
		return CDW_ERROR;
	}
	rv = set_menu_sub(dropdown->menu, dropdown->menu_window_sub);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to set menu subwindow, errno = \"%s\"\n", cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}

	/* remember that post_menu() displays menu in its subwindow, not on the
	   screen - you have to call refresh() to make menu visible to user */
	rv = post_menu(dropdown->menu);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to post menu, errno = \"%s\"\n", cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}

	dropdown->visible = true;

	return CDW_OK;
}





/**
   \brief Redraw window in which a dropdown is displayed

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   'Show' a dropdown simply by (re)drawing and refreshing window, in which
   dropdown is displayed (not the parent window, but the window used in
   implementation of the widget). Such dropdown now displays all items available.
   TODO: this is not the best way to "show" dropdown.

   \param dropdown - dropdown to show

   \return CDW_OK
*/
cdw_rv_t cdw_dropdown_expand(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot show NULL dropdown\n");
	cdw_assert (dropdown->menu_window, "ERROR: cannot show dropdown with NULL window\n");
	cdw_assert (dropdown->menu_window_sub, "ERROR: cannot show dropdown with NULL subwindow\n");

	/* move highlight to current item */
	menu_driver(dropdown->menu, REQ_FIRST_ITEM);
	for (int i = 0; i < dropdown->current_item_ind && i < dropdown->n_items; i++) {
		menu_driver(dropdown->menu, REQ_DOWN_ITEM);
	}

	redrawwin(dropdown->menu_window);
	redrawwin(dropdown->menu_window_sub);
	wrefresh(dropdown->menu_window);
	wrefresh(dropdown->menu_window_sub);
	return CDW_OK;
}





/**
   \brief Collapse given dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Collapse expanded \p dropdown simply by (re)drawing and refreshing its
   parent window. "Minimized" dropdown with hidden list is still visible,
   but it only displays selected item.

   TODO: This may not be the best way to collapse a dropdown.

   This function does not display currently selected item in place where
   dropdown is displayed.

   \param dropdown - dropdown to collapse

   \return CDW_OK
*/
cdw_rv_t cdw_dropdown_collapse(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot hide NULL dropdown\n");
	cdw_assert (dropdown->parent, "ERROR: cannot hide dropdown with NULL parent\n");

	redrawwin(dropdown->parent);
	wrefresh(dropdown->parent);

	return CDW_OK;
}





/**
   \brief Change visibility of a dropdown to 'invisible'

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Make a \p dropdown invisible. The \p dropdown won't be seen
   in user interface, and won't be accessible through TAB key.

   \param dropdown - dropdown to make invisible.
*/
void cdw_dropdown_make_invisible(CDW_DROPDOWN *dropdown)
{
	mvwhline(dropdown->parent, dropdown->begin_y, dropdown->begin_x, ' ', dropdown->width + 1);
	wrefresh(dropdown->parent);

	dropdown->visible = false;

	return;
}





/**
   \brief Change visibility of a dropdown to 'visible'

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Make a \p dropdown visible. The \p dropdown will be seen in user
   interface, and will be accessible through TAB key.

   \param dropdown - dropdown to make visible.
*/
void cdw_dropdown_make_visible(CDW_DROPDOWN *dropdown)
{
	dropdown->visible = true;
	cdw_dropdown_display_current_item_reverse(dropdown, false);

	return;
}





/**
   \brief Control keyboard input when focus is on dropdown - top level function

   This driver should be used only as (dropdown variable)->driver().

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   This function controls keyboard input while a \p dropdown has
   keyboard focus.

   Since this function will be used in some kind of dispatcher calling
   drivers for widgets of different types, we need to pass \p dropdown
   as void pointer.

   \param dd - dropdown that has focus
   \param dummy - pointer to additional data; currently unused

   \return CDW_KEY_ENTER if user pressed ENTER in expanded dropdown
   \return CDW_KEY_ESCAPE if user pressed ESCAPE, Q or q in expanded dropdown
   \return if any other key has been pressed the function returns the key
*/
int cdw_dropdown_driver(CDW_DROPDOWN *dd, __attribute__((unused)) void *dummy)
{
	cdw_assert (dd, "ERROR: NULL pointer to dropdown\n");
	cdw_assert (dd->widget_id == CDW_WIDGET_ID_DROPDOWN, "ERROR: this is not a dropdown widget\n");
	cdw_assert (dd->parent, "ERROR: cannot control dropdown with NULL parent\n");

	int key = 'a'; /* Safe initial value. */

	while (1) {
		key = wgetch(dd->parent);

		if (key == CDW_KEY_ENTER) {
			cdw_dropdown_expand(dd);
			int expanded_key = cdw_dropdown_expanded_driver(dd);
			cdw_dropdown_collapse(dd);

			if (expanded_key == CDW_KEY_ENTER) {
				/* when dropdown is closed (hidden), it should remain
				   highlighted, so second parameter is true */
				cdw_dropdown_display_current_item_reverse(dd, true);

				/* Call callback function, let it know which item (by
				   ID) has been selected. */
				if (dd->on_select_callback) {
					dd->on_select_callback(NULL, &(dd->items[dd->current_item_ind]->id));
				}
			}
		} else if (key == CDW_KEY_TAB || key == KEY_BTAB
			   || key == KEY_DOWN || key == KEY_UP) {

			/* Inter-widget movement key. Will be handled
			   by parent of the widget. */
			break;
		} else if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {

			key = CDW_KEY_ESCAPE;
			break;
		} else {
			/* Some other key that may be handled only by
			   parent of the widget. It may happen that
			   parent is configured to handle special keys
			   like F2 or other keys. */
			break;
		}
	}

	redrawwin(dd->parent);
	wrefresh(dd->parent);

	return key;
}





/**
   \brief React to keys pressed by user when given dropdown is expanded (active)

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   React to following keys: HOME, END, DOWN, UP, ENTER, ESCAPE, Q, q, when
   given dropdown is active/focused.

   Caller of this function must make sure that \p dropdown is valid.

   \param dropdown - active dropdown, on which user operates

   \return CDW_KEY_ENTER if user pressed ENTER,
   \return CDW_KEY_ESCAPE if user pressed ESCAPE, Q or q
*/
int cdw_dropdown_expanded_driver(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot control NULL dropdown\n");

	/* a bit customized, but otherwise standard ncurses menu driver code */
	int key = 'a';
	while ((key = wgetch(dropdown->parent)) != ERR) {
		switch (key) {
		        case CDW_KEY_ESCAPE:
			/* including 'q' and 'Q', as suggested by RogerX */
		        case 'q':
		        case 'Q':
				return CDW_KEY_ESCAPE;
			case KEY_HOME:
				menu_driver(dropdown->menu, REQ_FIRST_ITEM);
				break;
			case KEY_END:
				menu_driver(dropdown->menu, REQ_LAST_ITEM);
				break;
			case KEY_DOWN:
				menu_driver(dropdown->menu, REQ_DOWN_ITEM);
				break;
			case KEY_UP:
				menu_driver(dropdown->menu, REQ_UP_ITEM);
				break;
			case CDW_KEY_ENTER:
			{
				ITEM *item = current_item(dropdown->menu);
				cdw_assert (item, "ERROR: current item is NULL\n");
				/* remember index of selected item */
				dropdown->current_item_ind = item_index(item);

				return CDW_KEY_ENTER;
			}

			default: /* other (meaningless in this context) keys */
				break;
		}

		redrawwin(dropdown->menu_window_sub);
		wrefresh(dropdown->menu_window_sub);
	}

	return CDW_KEY_ESCAPE;
}





/**
   \brief Destroy given dropdown, deallocate its resources

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   The function sets *dropdown to NULL.

   \param dropdown - pointer to dropdown that you want to destroy

   \return CDW_OK
*/
cdw_rv_t cdw_dropdown_delete(CDW_DROPDOWN **dropdown)
{
	cdw_assert (dropdown,
		    "ERROR: you passed NULL pointer to function\n");

	if (!dropdown) {
		/* probably called the function when other function that
		   creates dropdown failed to allocate initial memory for
		   dropdown - hence null pointer */
		cdw_vdm ("WARNING: you passed NULL dropdown to function\n");
	} else {
		cdw_dropdown_free(*dropdown);
		*dropdown = (CDW_DROPDOWN *) NULL;
	}

	return CDW_OK;
}





/**
   \brief Free given dropdown, with all its resources

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Function de-allocates memory pointed to by \p dropdown,
   but does not set \p dropdown pointer to NULL.
   The function is usually called by cdw_dropdown_delete().

   \param dropdown - dropdown that you want to free
*/
void cdw_dropdown_free(CDW_DROPDOWN *dropdown)
{
	if (!dropdown) {
		/* probably called the function when other function that
		   creates dropdown failed to allocate initial memory for
		   dropdown - hence null pointer */
		cdw_vdm ("WARNING: you passed NULL dropdown to function\n");
		return;
	}

	if (dropdown->menu) {
		unpost_menu(dropdown->menu);
		free_menu(dropdown->menu);
		dropdown->menu = (MENU *) NULL;

		for (int j = 0; j < dropdown->n_items; j++){
			free_item(dropdown->menu_items[j]);
			dropdown->menu_items[j] = (ITEM *) NULL;
		}

		free(dropdown->menu_items);
		dropdown->menu_items = (ITEM **) NULL;
	}

	for (int i = 0; i < dropdown->n_items; i++) {
		if (dropdown->items[i]) {
			if (dropdown->items[i]->label) {
				free(dropdown->items[i]->label);
				dropdown->items[i]->label = (char *) NULL;
			} else {
				cdw_vdm ("WARNING: menu item label #%d is null\n", i);
			}

			free(dropdown->items[i]);
			dropdown->items[i] = (cdw_id_label_t *) NULL;
		} else {
			cdw_vdm ("WARNING: menu item #%d is null\n", i);
		}
	}
	free(dropdown->items);
	dropdown->items = (cdw_id_label_t **) NULL;

	delwin(dropdown->menu_window_sub);
	dropdown->menu_window_sub = (WINDOW *) NULL;

	delwin(dropdown->menu_window);
	dropdown->menu_window = (WINDOW *) NULL;

	free(dropdown);

	return;
}





/**
   \brief Return index of currently selected (active) dropdown item

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   \param dropdown - dropdown from which caller wants to get index of current item

   \return index of currently selected item
*/
int cdw_dropdown_get_current_item_ind(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot get index of item in NULL dropdown\n");

	return dropdown->current_item_ind;
}





/**
   \brief Display item currently selected in dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Force displaying in parent window current item of given \p dropdown.
   The item is displayed in its regular color scheme.

   \param dropdown - dropdown for which you want to display current item
*/
void cdw_dropdown_display_current_item(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot display current item in NULL dropdown\n");
	if (dropdown->visible) {
		cdw_dropdown_display_current_item_reverse(dropdown, false);
	}

	return;
}





/**
   \brief Display current item of a dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Force displaying dropdown's current item. You can control mode
   of displaying through \p reverse argument. When the argument
   is true, the item is displayed in reverse colors, so the dropdown
   looks like (but does not behave like!) it has focus.

   \param dropdown - dropdown for which you want to display current item
   \param reverse - mode of display
*/
void cdw_dropdown_display_current_item_reverse(CDW_DROPDOWN *dropdown, bool reverse)
{
	cdw_assert (dropdown, "ERROR: cannot display item in NULL dropdown\n");
	if (!dropdown->visible) {
		return;
	}

	cdw_assert (dropdown, "ERROR: cannot show current item of NULL dropdown\n");
	cdw_assert (dropdown->parent,  "ERROR: cannot show current item of dropdown with NULL parent\n");

	/* first erase old content */
	if (reverse) {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors) | A_REVERSE);
	} else {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors));
	}
	mvwhline(dropdown->parent, dropdown->begin_y, dropdown->begin_x, ' ', dropdown->width);


	/* then opening and closing bracket */
	mvwprintw(dropdown->parent, dropdown->begin_y, dropdown->begin_x, "[");
	mvwprintw(dropdown->parent, dropdown->begin_y, dropdown->begin_x + dropdown->width - 1, "]");

	/* 'real' content */
	mvwprintw(dropdown->parent, dropdown->begin_y, dropdown->begin_x + 1, dropdown->items[dropdown->current_item_ind]->label);

	/* and now fancy, bolded arrow by the dropdown */
	if (reverse) {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors) | A_REVERSE | A_BOLD);
	} else {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors) | A_BOLD);
	}
	mvwprintw(dropdown->parent, dropdown->begin_y, dropdown->begin_x + dropdown->width - 2, "v");

	(void) wattrset(dropdown->parent, A_NORMAL | COLOR_PAIR(dropdown->colors));

	wrefresh(dropdown->parent);

	return;
}





/**
   \brief Highlight given dropdown to mark that it has focus

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Use this function to change appearance of a non-expanded \p dropdown
   to make it look like it has now focus. After that the \p dropdown
   may be expanded with appropriate keyboard key.

   \param dropdown - dropdown that you want to focus on
*/
void cdw_dropdown_focus(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot focus NULL dropdown\n");
	if (dropdown->visible) {
		cdw_dropdown_display_current_item_reverse(dropdown, true);
	} else {
		cdw_vdm ("ERROR: called the function for invisible dropdown\n");
	}

	return;
}





/**
   \brief Un-highlight given dropdown to mark that it has lost focus

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Use this function to undo changes of appearance of a non-expanded
   \p dropdown made by cdw_dropdown_focus() -  to make it look like it
   has lost focus.  After that the \p dropdown cannot be expanded with
   appropriate keyboard key - until it gets focus again.

   \param dropdown - dropdown that you want to unfocus
*/
void cdw_dropdown_unfocus(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot unfocus NULL dropdown\n");
	if (dropdown->visible) {
		cdw_dropdown_display_current_item_reverse(dropdown, false);
	} else {
		cdw_vdm ("WARNING: called the function for invisible dropdown\n");
	}

	return;
}





/**
   \brief Set one of items of dropdown as selected/current value

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Use item's index \p ind to select one of dropdown's item and
   set it as currently selected/highlighted.

   \param dropdown - dropdown from which you want to select item
   \param ind - index of item that you want to select

   \return CDW_OK on success
   \return CDW_ERROR on errors (e.g. when there is no item with given index)
*/
cdw_rv_t cdw_dropdown_set_current_item_by_ind(CDW_DROPDOWN *dropdown, int ind)
{
	cdw_assert (dropdown, "ERROR: cannot set current item for NULL dropdown\n");

	if (ind < dropdown->n_items) {
		dropdown->current_item_ind = ind;
		cdw_dropdown_display_current_item(dropdown);

		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: index (%d) is larger than number of items (%d)\n", ind, dropdown->n_items);
		return CDW_ERROR;
	}
}





/**
   \brief Set one of items in dropdown as selected/current value

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Use item's \p id to select one of dropdown's item and set it
   as currently selected/highlighted.

   \param dropdown - dropdown from which you want to select item
   \param id - id of item that you want to select

   \return CDW_OK on success
   \return CDW_ERROR on errors (e.g. when there is no item with given id)
*/
cdw_rv_t cdw_dropdown_set_current_item_by_id(CDW_DROPDOWN *dropdown, cdw_id_t id)
{
	cdw_assert (dropdown, "ERROR: passing null dropdown to function\n");

	for (int i = 0; i < dropdown->n_items; i++) {
		if (dropdown->items[i]->id == id) {
			dropdown->current_item_ind = i;
			return CDW_OK;
		}
	}
	cdw_vdm ("ERROR: failed to set current item index by id, id = %lld\n", id);
	return CDW_ERROR;
}





/**
   \brief Get id associated with current item in a dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Get id of item currently selected/highlighted in given \p dropdown.

   \param dropdown - dropdown that you want to query

   \return id of current item
*/
cdw_id_t cdw_dropdown_get_current_item_id(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot set current item id from NULL dropdown\n");

	return dropdown->items[dropdown->current_item_ind]->id;
}





/**
   \brief Get label associated with current item in a dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Get label of item currently selected/highlighted in given \p dropdown.
   Returned pointer is owned by a dropdown, as long as the dropdown exists.

   \param dropdown - dropdown that you want to query

   \return label of current item
*/
const char *cdw_dropdown_get_current_item_label(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown, "ERROR: cannot set current item id from NULL dropdown\n");

	const char *label = dropdown->items[dropdown->current_item_ind]->label;
	cdw_assert (label, "ERROR: current label is NULL\n");
	return label;
}





/**
   \brief Get label of dropdown item specified by index

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Get label of item pointed to by specified index \p ind.
   Returned pointer is owned by a dropdown, as long as the dropdown exists.

   \param dropdown - dropdown that you want to query
   \param ind - index of an item

   \return label of item specified by index
*/
const char *cdw_dropdown_get_label_by_ind(CDW_DROPDOWN *dropdown, int ind)
{
	cdw_assert (dropdown, "ERROR: cannot set current item id from NULL dropdown\n");
	cdw_assert (ind < dropdown->n_items, "ERROR: index (%d) larger than number of items (%d)\n",
		    ind, dropdown->n_items);

	char *label = dropdown->items[ind]->label;
	cdw_assert (label, "ERROR: label #%d is NULL\n", ind);

	return label;
}





/**
   \brief Get label of dropdown item specified by id

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Get label of item indicated by \p id.
   Returned pointer is owned by a dropdown, as long as the dropdown exists.

   \param dropdown - dropdown that you want to query
   \param id - id of an item

   \return label of item specified by id on success
   \return NULL on failure (e.g. when there is no such id in given dropdown)
*/
const char *cdw_dropdown_get_label_by_id(CDW_DROPDOWN *dropdown, cdw_id_t id)
{
	cdw_assert (dropdown, "ERROR: cannot set current item id from NULL dropdown\n");

	for (int i = 0; i < dropdown->n_items; i++) {
		if (dropdown->items[i]->id == id) {
			char *label = dropdown->items[i]->label;
			cdw_assert (label, "ERROR: item with id %lld has null label\n", id);
			return label;
		}
	}

	cdw_vdm ("ERROR: can't find item for id %lld\n", id);

	return (char *) NULL;
}





/**
   \brief Wrapper function for creating a dropdown

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Wrapper for a set of functions creating a dropdown. Useful when you
   already have a table of 'label-id' pairs - you don't have to explicitly
   call cdw_dropdown_add_item() for every new item. You don't have to
   explicitly call cdw_dropdown_finalize() either.

   Color scheme of the dropdown is hardwired as CDW_COLORS_DIALOG.

   \param parent - parent window
   \param begin_y - y coordinate of a dropdown in parent window
   \param begin_x - x coordinate of a dropdown in parent window
   \param width - width of a dropdown
   \param n_items - number of items in 'items' table
   \param items - table of 'label-id' pairs, with size specified by 'n_items' parameter

   \return pointer to new and finalized dropdown on success
   \return NULL pointer on failure
*/
CDW_DROPDOWN *cdw_dropdown_maker_wrapper(WINDOW *parent, int begin_y, int begin_x, int width, int n_items, cdw_id_clabel_t *items)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(parent, begin_y, begin_x, width, n_items, CDW_COLORS_DIALOG);
	if (!dropdown) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	dropdown->on_select_callback = (cdw_form_widget_function_t) NULL;

	for (int i = 0; i < n_items; i++) {
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, items[i].id, _(items[i].label));
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item #%d (id = %lld, label = \"%s\"\n",
				 i, items[i].id, items[i].label);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}
	return dropdown;
}
