//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Tool/GroupBoxCollapser.h
//! @brief     Defines class GroupBoxCollapser
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_TOOL_GROUPBOXCOLLAPSER_H
#define BORNAGAIN_GUI_VIEW_TOOL_GROUPBOXCOLLAPSER_H

#include <QObject>

class QToolButton;
class QGroupBox;
class QHBoxLayout;
class QAction;

//! Add-on to group boxes to make them collapsible.
//!
//! Also adds the possibility to show toolbuttons or widgets in the group box's title.
//! This add-on takes the layout (plus the contained widgets) of a given group box and moves it into
//! a newly created content area widget. A title widget will be added, and the original group box
//! title will be hidden.
//!
//! To support style sheets, the custom bool property "collapsible" will be added to the group box
//! an set to true.
class GroupBoxCollapser : public QObject {
    Q_OBJECT
public:
    static GroupBoxCollapser* installIntoGroupBox(QGroupBox* groupBox, bool expanded = true);
    static GroupBoxCollapser* findInstalledCollapser(QGroupBox* groupBox);

    //! Set the title of the group box. Do not use the method groupBox->setTitle() any more once the
    //! add-on is installed.
    void setTitle(const QString& title);

    //! Add a tool button to the title bar, connected to the given action
    void addAction(QAction* action);

    //! Add a widget to the title bar.
    void addWidget(QWidget* widget);

    //! The content area, to which the found group box content has been moved when the add-on has
    //! been installed.
    QWidget* contentArea() const;

    //! Expand/collapse the content area
    void setExpanded(bool expanded = true);

signals:
    void toggled(bool b);

private:
    GroupBoxCollapser(QGroupBox* groupBox);
    void toggle(bool checked);

    QWidget* m_titleWidget;      //!< widget used to present the new groupbox title
    QHBoxLayout* m_titleLayout;  //!< layout in the title widget
    QToolButton* m_toggleButton; //!< button to toggle between collapsed/expanded
    QWidget* m_contentArea;      //!< widget to where the original group box content has been moved
};


#endif // BORNAGAIN_GUI_VIEW_TOOL_GROUPBOXCOLLAPSER_H
