//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/SampleItem.h
//! @brief     Defines class SampleItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_SAMPLE_SAMPLEITEM_H
#define BORNAGAIN_GUI_MODEL_SAMPLE_SAMPLEITEM_H

#include "Base/Types/OwningVector.h"
#include "GUI/Model/Descriptor/VectorProperty.h"
#include "GUI/Model/Sample/Item3D.h"
#include "GUI/Model/Sample/MaterialModel.h"
#include <QString>
#include <QUuid>
#include <QVector>
#include <QXmlStreamReader>

class ItemWithMaterial;
class LayerItem;

class SampleItem : public virtual Item3D {
public:
    SampleItem();
    ~SampleItem();

    void initFrom(const SampleItem* other);

    QVector<ItemWithMaterial*> itemsWithMaterial() const;

    void addStandardMaterials();

    QString sampleName() const { return m_name; }
    void setSampleName(const QString& s) { m_name = s; }

    QString description() const { return m_description; }
    void setDescription(const QString& s) { m_description = s; }

    DoubleProperty& crossCorrLength() { return m_crossCorrelationLength; }
    const DoubleProperty& crossCorrLength() const { return m_crossCorrelationLength; }
    void setCrossCorLength(double d) { m_crossCorrelationLength.setValue(d); }

    VectorProperty& externalField() { return m_externalField; }
    const VectorProperty& externalField() const { return m_externalField; }
    void setExternalField(const R3& r) { m_externalField.setR3(r); }

    QVector<LayerItem*> layerItems() const;

    //! Creates and inserts a layer at given index.
    //!
    //! No properties etc. have been initialized; this has to be done by the caller.
    //! If index = -1, create a layer at the end of the list.
    LayerItem* createLayerItemAt(int index = -1);

    void updateTopBottom();
    void removeLayer(LayerItem* item);
    void moveLayer(LayerItem* item, LayerItem* aboveThisLayer);

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    MaterialModel& materialModel() { return m_materials; }
    const MaterialModel& materialModel() const { return m_materials; }

    bool isExpandInfo() const { return m_expandInfo; }
    void setExpandInfo(bool b) { m_expandInfo = b; }

private:
    QString m_name;
    QString m_description;
    DoubleProperty m_crossCorrelationLength;
    VectorProperty m_externalField;
    OwningVector<LayerItem> m_layers;
    MaterialModel m_materials;
    bool m_expandInfo = true;
};

#endif // BORNAGAIN_GUI_MODEL_SAMPLE_SAMPLEITEM_H
