/*
 * Copyright (c) 2022-2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void sme2_gemv_u8qa_dot_16VL (
    const uint8_t *A_ptr, const uint8_t *B_ptr, uint8_t *output_ptr,
    size_t N, size_t K,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        const uint8_t *B_ptr = {};
        size_t output_offset = {};
        unsigned int input_initial_col = {};
    } ka;

    unsigned long flags=0;
    ka.B_ptr = B_ptr;
    if (qp->c_offset > qp->minval) {
        flags |= 0x20;
    }
    __asm__ __volatile__(
      "ptrue p8.b\n"
      ".inst 0xd503477f  // SMSTART ZA\n"
      "cntw x28, ALL, MUL #4\n"
      "add x27, %x[N], x28\n"
      "sub x27, x27, #0x1\n"
      "udiv x27, x27, x28\n"
      "add x22, x27, #0x3\n"
      "and x22, x22, #0xfffffffffffffffc\n"
      "mul x22, x22, x28\n"
      "mov x9, #0x0\n"
      "mov x26, %x[B_ptr]\n"
      "mov x25, %x[output_ptr]\n"
      "ptrue p2.b\n"
      ".inst 0x25207810  // ptrue pn8.b\n"
      "mul x22, x22, %x[K]\n"
      "mov x21, #0x1\n"
      "1:"  // RHS size check loop
      "cmp x22, #0x200000\n"
      "blt 2f\n"
      "tbnz x22, #0, 3f\n"
      "lsr x22, x22, #0x1\n"
      "lsl x21, x21, #0x1\n"
      "b 1b\n"
      "2:"  // RHS do prefetch
      "lsl x20, x22, #0x26\n"
      "sub x21, x21, #0x1\n"
      "lsl x21, x21, #0x16\n"
      "orr x22, x22, x20\n"
      "orr x22, x22, x21\n"
      ".inst 0xf8b64b5a  // rprfm pldonce, x22, [x26]\n"
      "3:"  // RHS prefetch exit
      "mov x24, %x[col_bias]\n"
      "mov z28.s, #0x0\n"
      "mov z29.b, #0x1\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "4:"  // Column loop
      "cmp x27, #0x4\n"
      "bge 34f\n"
      "cmp x27, #0x2\n"
      "bgt 24f\n"
      "beq 14f\n"
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "mov x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 5f\n"
      ".inst 0xa040c300  // ld1w { z0.s-z3.s }, pn8.b/Z, [x24]\n"
      ".inst 0xc0042c00  // mova za.d[x9, #0], { z0.d-z3.d }\n"
      "b 6f\n"
      "5:"  // Width 1: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "6:"  // Width 1: setup done
      "cmp x22, #0x10\n"
      "ble 9f\n"
      "7:"  // Width 1: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0408355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b2b0  // udot za.s[x9, 0], { z20.b-z23.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b5b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b9b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151bdb0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "tbnz %x[flags], #31, 8f\n"
      "udot z28.s, z1.b, z29.b\n"
      "8:"  // Width 1: Multiply loop: unique 1: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 7b\n"
      "9:"  // Width 1: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xc151b1b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      "ble 10f\n"
      ".inst 0xa0408355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151b6b0  // udot za.s[x9, 0], { z20.b-z23.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      "ble 10f\n"
      ".inst 0xa0408349  // ldnt1b { z8.b-z11.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151b930  // udot za.s[x9, 0], { z8.b-z11.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      "ble 10f\n"
      ".inst 0xa0408349  // ldnt1b { z8.b-z11.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151bd30  // udot za.s[x9, 0], { z8.b-z11.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "10:"  // Width 1: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 11f\n"
      "udot z28.s, z1.b, z29.b\n"
      "11:"  // Width 1: Multiply loop: unique 2: skip row sum
      "tbnz %x[flags], #31, 12f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z26.s }, p2/Z, [x21]\n"
      "neg z26.s, p2/M, z26.s\n"
      "whilelt p0.s, XZR, x20\n"
      "uaddv d28, p0, z28.s\n"
      "mov z28.s, z28.s[0]\n"
      "mul z28.s, p2/M, z28.s, z26.s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "12:"  // Width 1: skip row sum fixup
      ".inst 0xc0904b80  // addha za0.s, p2/M, p2/M, z28.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z1.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904b81  // addha za1.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z7.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904b82  // addha za2.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z2.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904b83  // addha za3.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      ".inst 0xc0062c0c  // mova { z12.d-z15.d }, za.d[x9, #0]\n"
      ".inst 0xc1a1ac0c  // sqdmulh { z12.s-z15.s }, { z12.s-z15.s }, z1.s\n"
      ".inst 0xc1a7aa2c  // srshl { z12.s-z15.s }, { z12.s-z15.s }, z7.s\n"
      "ld1rw { z30.s }, p2/Z, [x20]\n"
      ".inst 0xc1a2ab0c  // add { z12.s-z15.s }, { z12.s-z15.s }, z2.s\n"
      ".inst 0xc1bece0c  // sclamp { z12.s-z15.s }, z16.s, z30.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z19.h, z14.h, z15.h\n"
      "uzp1 z12.b, z12.b, z19.b\n"
      "st1b { z12.b }, p1, [x25]\n"
      "addvl x25, x25, #1\n"
      "13:"  // Width 1: Output done
      "b 44f\n"
      "14:"  // Width 2
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "sub x20, %x[N], x28\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 15f\n"
      ".inst 0xa040c310  // ld1w { z16.s-z19.s }, pn8.b/Z, [x24]\n"
      ".inst 0xc0042e00  // mova za.d[x9, #0], { z16.d-z19.d }\n"
      ".inst 0xa041c318  // ld1w { z24.s-z27.s }, pn8.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xc0042f01  // mova za.d[x9, #1], { z24.d-z27.d }\n"
      "b 16f\n"
      "15:"  // Width 2: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "16:"  // Width 2: setup done
      "cmp x22, #0x10\n"
      "ble 19f\n"
      "17:"  // Width 2: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b1b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[0]\n"
      ".inst 0xa0418359  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b331  // udot za.s[x9, 1], { z24.b-z27.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b630  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[1]\n"
      ".inst 0xa0418355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b6b1  // udot za.s[x9, 1], { z20.b-z23.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b9b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[2]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b9b1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa0408345  // ldnt1b { z4.b-z7.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151bcb0  // udot za.s[x9, 0], { z4.b-z7.b }, z1.b[3]\n"
      ".inst 0xa0418349  // ldnt1b { z8.b-z11.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151bd31  // udot za.s[x9, 1], { z8.b-z11.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "tbnz %x[flags], #31, 18f\n"
      "udot z28.s, z1.b, z29.b\n"
      "18:"  // Width 2: Multiply loop: unique 3: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 17b\n"
      "19:"  // Width 2: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0408359  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x26]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xc151b330  // udot za.s[x9, 0], { z24.b-z27.b }, z1.b[0]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b231  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      "ble 20f\n"
      ".inst 0xa0408355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151b6b0  // udot za.s[x9, 0], { z20.b-z23.b }, z1.b[1]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b5b1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      "ble 20f\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151b9b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[2]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b9b1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      "ble 20f\n"
      ".inst 0xa0408359  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151bf30  // udot za.s[x9, 0], { z24.b-z27.b }, z1.b[3]\n"
      ".inst 0xa0418349  // ldnt1b { z8.b-z11.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151bd31  // udot za.s[x9, 1], { z8.b-z11.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "20:"  // Width 2: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 21f\n"
      "udot z28.s, z1.b, z29.b\n"
      "21:"  // Width 2: Multiply loop: unique 4: skip row sum
      "tbnz %x[flags], #31, 22f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      "neg z16.s, p2/M, z16.s\n"
      "whilelt p0.s, XZR, x20\n"
      "uaddv d28, p0, z28.s\n"
      "mov z28.s, z28.s[0]\n"
      "mul z28.s, p2/M, z28.s, z16.s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "22:"  // Width 2: skip row sum fixup
      ".inst 0xc0904b80  // addha za0.s, p2/M, p2/M, z28.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z6.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904b81  // addha za1.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904b82  // addha za2.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z9.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904b83  // addha za3.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      ".inst 0xc0062c18  // mova { z24.d-z27.d }, za.d[x9, #0]\n"
      ".inst 0xc1a6ac18  // sqdmulh { z24.s-z27.s }, { z24.s-z27.s }, z6.s\n"
      ".inst 0xc0062c20  // mova { z0.d-z3.d }, za.d[x9, #1]\n"
      ".inst 0xc1a6ac00  // sqdmulh { z0.s-z3.s }, { z0.s-z3.s }, z6.s\n"
      ".inst 0xc1a5aa38  // srshl { z24.s-z27.s }, { z24.s-z27.s }, z5.s\n"
      "ld1rw { z21.s }, p2/Z, [x20]\n"
      ".inst 0xc1a5aa20  // srshl { z0.s-z3.s }, { z0.s-z3.s }, z5.s\n"
      ".inst 0xc1a9ab18  // add { z24.s-z27.s }, { z24.s-z27.s }, z9.s\n"
      ".inst 0xc1a9ab00  // add { z0.s-z3.s }, { z0.s-z3.s }, z9.s\n"
      ".inst 0xc1b5ce18  // sclamp { z24.s-z27.s }, z16.s, z21.s\n"
      ".inst 0xc1b5ce00  // sclamp { z0.s-z3.s }, z16.s, z21.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z9.h, z26.h, z27.h\n"
      "uzp1 z0.h, z0.h, z1.h\n"
      "uzp1 z26.h, z2.h, z3.h\n"
      "uzp1 z24.b, z24.b, z9.b\n"
      "st1b { z24.b }, p2, [x25]\n"
      "uzp1 z0.b, z0.b, z26.b\n"
      "st1b { z0.b }, p1, [x25, #1, MUL VL]\n"
      "addvl x25, x25, #2\n"
      "23:"  // Width 2: Output done
      "b 44f\n"
      "24:"  // Width 3
      "mov x20, #0x2\n"
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 25f\n"
      ".inst 0xa040c310  // ld1w { z16.s-z19.s }, pn8.b/Z, [x24]\n"
      ".inst 0xc0042e00  // mova za.d[x9, #0], { z16.d-z19.d }\n"
      ".inst 0xa041c30c  // ld1w { z12.s-z15.s }, pn8.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xc0042d81  // mova za.d[x9, #1], { z12.d-z15.d }\n"
      ".inst 0xa042c318  // ld1w { z24.s-z27.s }, pn8.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xc0042f02  // mova za.d[x9, #2], { z24.d-z27.d }\n"
      "b 26f\n"
      "25:"  // Width 3: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "26:"  // Width 3: setup done
      "cmp x22, #0x10\n"
      "ble 29f\n"
      "27:"  // Width 3: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b230  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[0]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b231  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[0]\n"
      ".inst 0xa042834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b1b2  // udot za.s[x9, 2], { z12.b-z15.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b5b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[1]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b5b1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[1]\n"
      ".inst 0xa0428355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b6b2  // udot za.s[x9, 2], { z20.b-z23.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa0408349  // ldnt1b { z8.b-z11.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b930  // udot za.s[x9, 0], { z8.b-z11.b }, z1.b[2]\n"
      ".inst 0xa0418345  // ldnt1b { z4.b-z7.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b8b1  // udot za.s[x9, 1], { z4.b-z7.b }, z1.b[2]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151ba32  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa0408359  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151bf30  // udot za.s[x9, 0], { z24.b-z27.b }, z1.b[3]\n"
      ".inst 0xa0418345  // ldnt1b { z4.b-z7.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151bcb1  // udot za.s[x9, 1], { z4.b-z7.b }, z1.b[3]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151be32  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "tbnz %x[flags], #31, 28f\n"
      "udot z28.s, z1.b, z29.b\n"
      "28:"  // Width 3: Multiply loop: unique 5: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 27b\n"
      "29:"  // Width 3: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0408355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xc151b2b0  // udot za.s[x9, 0], { z20.b-z23.b }, z1.b[0]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b231  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[0]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b232  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      "ble 30f\n"
      ".inst 0xa0408359  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151b730  // udot za.s[x9, 0], { z24.b-z27.b }, z1.b[1]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b5b1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[1]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b632  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      "ble 30f\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151ba30  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[2]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151ba31  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[2]\n"
      ".inst 0xa0428355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151bab2  // udot za.s[x9, 2], { z20.b-z23.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      "ble 30f\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151bdb0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[3]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151be31  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[3]\n"
      ".inst 0xa042834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151bdb2  // udot za.s[x9, 2], { z12.b-z15.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "30:"  // Width 3: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 31f\n"
      "udot z28.s, z1.b, z29.b\n"
      "31:"  // Width 3: Multiply loop: unique 6: skip row sum
      "tbnz %x[flags], #31, 32f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      "neg z16.s, p2/M, z16.s\n"
      "whilelt p0.s, XZR, x20\n"
      "uaddv d28, p0, z28.s\n"
      "mov z28.s, z28.s[0]\n"
      "mul z28.s, p2/M, z28.s, z16.s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "32:"  // Width 3: skip row sum fixup
      ".inst 0xc0904b80  // addha za0.s, p2/M, p2/M, z28.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z2.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904b81  // addha za1.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z1.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904b82  // addha za2.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z3.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904b83  // addha za3.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      ".inst 0xc0062c08  // mova { z8.d-z11.d }, za.d[x9, #0]\n"
      ".inst 0xc1a2ac08  // sqdmulh { z8.s-z11.s }, { z8.s-z11.s }, z2.s\n"
      ".inst 0xc0062c24  // mova { z4.d-z7.d }, za.d[x9, #1]\n"
      ".inst 0xc1a2ac04  // sqdmulh { z4.s-z7.s }, { z4.s-z7.s }, z2.s\n"
      ".inst 0xc0062c4c  // mova { z12.d-z15.d }, za.d[x9, #2]\n"
      ".inst 0xc1a2ac0c  // sqdmulh { z12.s-z15.s }, { z12.s-z15.s }, z2.s\n"
      ".inst 0xc1a1aa28  // srshl { z8.s-z11.s }, { z8.s-z11.s }, z1.s\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      ".inst 0xc1a1aa24  // srshl { z4.s-z7.s }, { z4.s-z7.s }, z1.s\n"
      ".inst 0xc1a1aa2c  // srshl { z12.s-z15.s }, { z12.s-z15.s }, z1.s\n"
      ".inst 0xc1a3ab08  // add { z8.s-z11.s }, { z8.s-z11.s }, z3.s\n"
      ".inst 0xc1a3ab04  // add { z4.s-z7.s }, { z4.s-z7.s }, z3.s\n"
      ".inst 0xc1a3ab0c  // add { z12.s-z15.s }, { z12.s-z15.s }, z3.s\n"
      ".inst 0xc1a0ce08  // sclamp { z8.s-z11.s }, z16.s, z0.s\n"
      ".inst 0xc1a0ce04  // sclamp { z4.s-z7.s }, z16.s, z0.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      ".inst 0xc1a0ce0c  // sclamp { z12.s-z15.s }, z16.s, z0.s\n"
      "uzp1 z18.h, z10.h, z11.h\n"
      "uzp1 z4.h, z4.h, z5.h\n"
      "uzp1 z17.h, z6.h, z7.h\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z16.h, z14.h, z15.h\n"
      "uzp1 z8.b, z8.b, z18.b\n"
      "st1b { z8.b }, p2, [x25]\n"
      "uzp1 z4.b, z4.b, z17.b\n"
      "st1b { z4.b }, p2, [x25, #1, MUL VL]\n"
      "uzp1 z12.b, z12.b, z16.b\n"
      "st1b { z12.b }, p1, [x25, #2, MUL VL]\n"
      "addvl x25, x25, #3\n"
      "33:"  // Width 3: Output done
      "b 44f\n"
      "34:"  // Width 4
      "mov x20, #0x3\n"
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 35f\n"
      ".inst 0xa040c314  // ld1w { z20.s-z23.s }, pn8.b/Z, [x24]\n"
      ".inst 0xc0042e80  // mova za.d[x9, #0], { z20.d-z23.d }\n"
      ".inst 0xa041c310  // ld1w { z16.s-z19.s }, pn8.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xc0042e01  // mova za.d[x9, #1], { z16.d-z19.d }\n"
      ".inst 0xa042c310  // ld1w { z16.s-z19.s }, pn8.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xc0042e02  // mova za.d[x9, #2], { z16.d-z19.d }\n"
      ".inst 0xa043c310  // ld1w { z16.s-z19.s }, pn8.b/Z, [x24, #0xc, MUL VL]\n"
      ".inst 0xc0042e03  // mova za.d[x9, #3], { z16.d-z19.d }\n"
      "addvl x24, x24, #16\n"
      "b 36f\n"
      "35:"  // Width 4: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "36:"  // Width 4: setup done
      "cmp x22, #0x10\n"
      "ble 39f\n"
      "37:"  // Width 4: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b230  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[0]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b231  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[0]\n"
      ".inst 0xa042834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b1b2  // udot za.s[x9, 2], { z12.b-z15.b }, z1.b[0]\n"
      ".inst 0xa043834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151b1b3  // udot za.s[x9, 3], { z12.b-z15.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151b630  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[1]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b5b1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[1]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b632  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[1]\n"
      ".inst 0xa0438351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151b633  // udot za.s[x9, 3], { z16.b-z19.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151ba30  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[2]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b9b1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[2]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151ba32  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[2]\n"
      ".inst 0xa0438351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151ba33  // udot za.s[x9, 3], { z16.b-z19.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151bdb0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[3]\n"
      ".inst 0xa041834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151bdb1  // udot za.s[x9, 1], { z12.b-z15.b }, z1.b[3]\n"
      ".inst 0xa0428359  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151bf32  // udot za.s[x9, 2], { z24.b-z27.b }, z1.b[3]\n"
      ".inst 0xa0438345  // ldnt1b { z4.b-z7.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151bcb3  // udot za.s[x9, 3], { z4.b-z7.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "tbnz %x[flags], #31, 38f\n"
      "udot z28.s, z1.b, z29.b\n"
      "38:"  // Width 4: Multiply loop: unique 7: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 37b\n"
      "39:"  // Width 4: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa040834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xc151b1b0  // udot za.s[x9, 0], { z12.b-z15.b }, z1.b[0]\n"
      ".inst 0xa0418359  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b331  // udot za.s[x9, 1], { z24.b-z27.b }, z1.b[0]\n"
      ".inst 0xa0428349  // ldnt1b { z8.b-z11.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b132  // udot za.s[x9, 2], { z8.b-z11.b }, z1.b[0]\n"
      ".inst 0xa0438351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151b233  // udot za.s[x9, 3], { z16.b-z19.b }, z1.b[0]\n"
      "addvl x26, x26, #16\n"
      "ble 40f\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151b630  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[1]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151b631  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[1]\n"
      ".inst 0xa042834d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151b5b2  // udot za.s[x9, 2], { z12.b-z15.b }, z1.b[1]\n"
      ".inst 0xa0438355  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151b6b3  // udot za.s[x9, 3], { z20.b-z23.b }, z1.b[1]\n"
      "addvl x26, x26, #16\n"
      "ble 40f\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xc151ba30  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[2]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151ba31  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[2]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151ba32  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[2]\n"
      ".inst 0xa0438351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151ba33  // udot za.s[x9, 3], { z16.b-z19.b }, z1.b[2]\n"
      "addvl x26, x26, #16\n"
      "ble 40f\n"
      ".inst 0xa0408351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26]\n"
      ".inst 0xc151be30  // udot za.s[x9, 0], { z16.b-z19.b }, z1.b[3]\n"
      ".inst 0xa0418351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc151be31  // udot za.s[x9, 1], { z16.b-z19.b }, z1.b[3]\n"
      ".inst 0xa0428351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc151be32  // udot za.s[x9, 2], { z16.b-z19.b }, z1.b[3]\n"
      ".inst 0xa0438351  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc151be33  // udot za.s[x9, 3], { z16.b-z19.b }, z1.b[3]\n"
      "addvl x26, x26, #16\n"
      "40:"  // Width 4: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 41f\n"
      "udot z28.s, z1.b, z29.b\n"
      "41:"  // Width 4: Multiply loop: unique 8: skip row sum
      "tbnz %x[flags], #31, 42f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      "neg z16.s, p2/M, z16.s\n"
      "whilelt p0.s, XZR, x20\n"
      "uaddv d28, p0, z28.s\n"
      "mov z28.s, z28.s[0]\n"
      "mul z28.s, p2/M, z28.s, z16.s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "42:"  // Width 4: skip row sum fixup
      ".inst 0xc0904b80  // addha za0.s, p2/M, p2/M, z28.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z11.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904b81  // addha za1.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z7.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904b82  // addha za2.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z6.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904b83  // addha za3.s, p2/M, p2/M, z28.s\n"
      "ld1rw { z3.s }, p2/Z, [x21]\n"
      ".inst 0xc0062c18  // mova { z24.d-z27.d }, za.d[x9, #0]\n"
      ".inst 0xc1abac18  // sqdmulh { z24.s-z27.s }, { z24.s-z27.s }, z11.s\n"
      ".inst 0xc0062c30  // mova { z16.d-z19.d }, za.d[x9, #1]\n"
      ".inst 0xc1abac10  // sqdmulh { z16.s-z19.s }, { z16.s-z19.s }, z11.s\n"
      ".inst 0xc0062c54  // mova { z20.d-z23.d }, za.d[x9, #2]\n"
      ".inst 0xc1abac14  // sqdmulh { z20.s-z23.s }, { z20.s-z23.s }, z11.s\n"
      ".inst 0xc0062c6c  // mova { z12.d-z15.d }, za.d[x9, #3]\n"
      ".inst 0xc1abac0c  // sqdmulh { z12.s-z15.s }, { z12.s-z15.s }, z11.s\n"
      ".inst 0xc1a7aa38  // srshl { z24.s-z27.s }, { z24.s-z27.s }, z7.s\n"
      "ld1rw { z31.s }, p2/Z, [x20]\n"
      ".inst 0xc1a7aa30  // srshl { z16.s-z19.s }, { z16.s-z19.s }, z7.s\n"
      ".inst 0xc1a7aa34  // srshl { z20.s-z23.s }, { z20.s-z23.s }, z7.s\n"
      ".inst 0xc1a7aa2c  // srshl { z12.s-z15.s }, { z12.s-z15.s }, z7.s\n"
      ".inst 0xc1a6ab18  // add { z24.s-z27.s }, { z24.s-z27.s }, z6.s\n"
      ".inst 0xc1a6ab10  // add { z16.s-z19.s }, { z16.s-z19.s }, z6.s\n"
      ".inst 0xc1a6ab14  // add { z20.s-z23.s }, { z20.s-z23.s }, z6.s\n"
      ".inst 0xc1a6ab0c  // add { z12.s-z15.s }, { z12.s-z15.s }, z6.s\n"
      ".inst 0xc1bfcc78  // sclamp { z24.s-z27.s }, z3.s, z31.s\n"
      ".inst 0xc1bfcc70  // sclamp { z16.s-z19.s }, z3.s, z31.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      ".inst 0xc1bfcc74  // sclamp { z20.s-z23.s }, z3.s, z31.s\n"
      ".inst 0xc1bfcc6c  // sclamp { z12.s-z15.s }, z3.s, z31.s\n"
      "uzp1 z25.h, z26.h, z27.h\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z18.h, z18.h, z19.h\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z17.h, z22.h, z23.h\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z30.h, z14.h, z15.h\n"
      "uzp1 z24.b, z24.b, z25.b\n"
      "st1b { z24.b }, p2, [x25]\n"
      "uzp1 z16.b, z16.b, z18.b\n"
      "st1b { z16.b }, p2, [x25, #1, MUL VL]\n"
      "uzp1 z20.b, z20.b, z17.b\n"
      "uzp1 z12.b, z12.b, z30.b\n"
      "st1b { z20.b }, p2, [x25, #2, MUL VL]\n"
      "st1b { z12.b }, p1, [x25, #3, MUL VL]\n"
      "addvl x25, x25, #4\n"
      "43:"  // Width 4: Output done
      "subs x27, x27, #0x4\n"
      "sub %x[N], %x[N], x28, LSL #2\n"
      "bgt 4b\n"
      "44:"  // Exit
      ".inst 0xd503467f  // SMSTOP\n"
      "ptrue p8.b\n"
      : [N] "+&r" (N), [flags] "+&r" (flags)
      : [A_ptr] "r" (A_ptr), [B_ptr] "r" (B_ptr), [K] "r" (K), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [output_ptr] "r" (output_ptr), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x9", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
