/* Chromium Unity integration extension
 * 
 *   Copyright 2012 Canonical Ltd.
 *
 *   This program is free software: you can redistribute it and/or modify it 
 *   under the terms of the GNU General Public License version 3, as published 
 *   by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful, but 
 *   WITHOUT ANY WARRANTY; without even the implied warranties of 
 *   MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 *   PURPOSE.  See the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along 
 *   with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

var background_page = (function () {
    // Native messaging support
    var HOST_ADDRESS = 'com.canonical.webapp.installer';
    var port = null;
    var host_callbacks = [];

    var portListener = function (msg) {
        callback = host_callbacks.pop();  
        if (callback === undefined) {
            return;
        }
        callback(msg);
    };

    var portDisconnecter = function () {
        console.log('UCX: port disconnected');
        port = null
    };

    var sendNativeMessage = function (msg, callback) {
        console.log("UCX: sendnativemessage: port=" + port);
        if (port === null) {
            port = chrome.runtime.connectNative(HOST_ADDRESS);
            port.onMessage.addListener(portListener);
            port.onDisconnect.addListener(portDisconnecter);
        }
        host_callbacks.push(callback || function(m) {});
        port.postMessage(msg);
    };

    // Wake server to catalogue webapps
    sendNativeMessage({"method":"ping"});


    /////////////////////////////////////////////////////////
    // 
    // Scafolding to keep track of data associated w/ infobar requests 
    // (Chromium's structure imposes some kind of state being maintained
    // in order to communicate data) 
    // 
    ////////////////////////////////////////////////////////
    // 
    // list of callback that are to be called asynchronously
    //  per tab. Used in the user integration/installation resquests context.
    // 
    // One thing to keep in mind is that one constraint, that bring some amount of
    //  'soundness' is that there is a hard limit (provided by the browser) of one infobar per tab.

    var user_infobar_request_callbacks = {};
    var addInfobarRequestCallbackFor = function (infobarRequestId, callback, message, details) {
        user_infobar_request_callbacks[infobarRequestId] = {
            callback: callback,
            message: message,
            details: details
        };
    };

    var getDataIfAnyFor = function (infobarRequestId) {
        if (user_infobar_request_callbacks[infobarRequestId] === undefined) {
            return "";
        }
        return {
            message: user_infobar_request_callbacks[infobarRequestId].message,
            details: user_infobar_request_callbacks[infobarRequestId].details
        };
    };

    var invokeAndRemoveCallbackIfAnyFor = function (infobarRequestId, arguments) {
        if (user_infobar_request_callbacks[infobarRequestId] === undefined) {
            return;
        }
        var callback = user_infobar_request_callbacks[infobarRequestId].callback;
        user_infobar_request_callbacks[infobarRequestId] = undefined;
        if (typeof(callback) === 'function') {
            callback(arguments);
        }
    };


    /**
     * Performs a match on the list of loaded integration scripts
     * given a url.
     *
     */
    var matchesIntegrationScripts = function (plugin, url, repo, windowInfos, callback) {
        sendNativeMessage({
                "method" : "url_loaded",
                "url" : url
            },
            function (response) {
                if (response.available) {
                    addInfobarRequestCallbackFor(
                        windowInfos.tabId,
                        function (result) {
                            if (result && result.integrate) {
                                sendNativeMessage({
                                    "method" : "install",
                                    "url" : url
                                })
                            } else {
                                sendNativeMessage({
                                    "method" : "dont_ask",
                                    "url" : url
                                })
                            }
                        },
                        chrome.i18n.getMessage("integration_copy", [ response.appName, response.appDomain ]),
                        null);

                    chrome.infobars.show({ tabId: windowInfos.tabId, path: "infobar.html" });
                }
            });
    }; 
   
   
   // {{{ main request handler
   
   /**
    * Handles & responds to content script requests.
    * 
    */
    var init_requested_stamps = {};
    var contentScriptsRequestHandler = function (request, sender, callback) {
        var handlers = {
            get_extension_settings: function (request, sender, callback) {
                var settings = {
                    logging: false,
                    incognito: sender.tab.incognito
                };
	 
                try {
                    if (window.localStorage) {
                       settings.logging = localStorage['logging'];
                    }
                }
                catch (e) {
                    console.log("Error while trying to retrieve logging information: " + str(e));
                }

	            callback (settings);
            },
            on_user_infobar_request_result: function (request, sender, callback) {
                invokeAndRemoveCallbackIfAnyFor (request.tabId, request);
            },
            init_requested: function (request, sender, callback) {
                sendNativeMessage({
                        "method" : "url_loaded",
                        "url" : request.options.url
                    },
                    function (response) {
                        if (response.available) {
                            addInfobarRequestCallbackFor(
                                sender.tab.id,
                                function (result) {
                                    if (result && result.integrate) {
                                        sendNativeMessage({
                                            "method" : "install",
                                            "url" : request.options.url
                                        })
                                    } else {
                                        sendNativeMessage({
                                            "method" : "dont_ask",
                                            "url" : request.options.url
                                        })
                                    }
                                },
                                chrome.i18n.getMessage("integration_copy", [ response.appName, response.appDomain ]),
                                null);

                            chrome.infobars.show({ tabId: windowInfos.tabId, path: "infobar.html" });
                        }
                    });
            }
        };
     
        // validate request
        if (!request  || !request.method) {
            callback({ error: "Invalid request structure" });
            return true;
        }
        if (!sender) {
            callback({ error: "Invalid sender" });
            return true;
        }

        if (typeof(request.method) != 'string' || request.method.length == 0) {
            callback({ error: "Invalid request method" });
            return true;
        }

        console.log('Got request: ' + request.method);

        var handler = handlers [request.method];
        if (handler !== undefined && typeof(handler) == 'function') {
            handler(request, sender, callback);
            return true;
        }
        return false;
    }

    // Main event handler and communication link
    //   w/ content scripts
    chrome.runtime.onMessage.addListener (contentScriptsRequestHandler);
    // }}}

    ///////////////////////////////////////////////////////////////////////
    // 
    // Window management related functions. In chromeless mode, we have specific
    //  rules for tab management to make webapps feel more "native" than plain
    //  web applications.
    // 
    ///////////////////////////////////////////////////////////////////////

    var onTabChanged = function (tabId, windowId, url) {
        var onInstalled = function (installed, packageName, appName, appDomain) { };

        console.log("onTabChanged: " + url);

        matchesIntegrationScripts(
            null, //plugin,
            url,
            null, // repo_,
            {
                windowId: windowId,
                tabId: tabId
            },
            onInstalled
        );
    };

    chrome.tabs.onCreated.addListener(
        function(tab) {
            if (tab && tab.url) {
                onTabChanged(tab.id, tab.windowId, tab.url);
            }
        }
    );

    chrome.tabs.onUpdated.addListener(
        function(tabId, changeInfo, tab) {
            console.log("onUpdated " + changeInfo.url);
            if (changeInfo && changeInfo.url) {
                onTabChanged(tabId, tab.windowId, changeInfo.url);
            }
        }
    );

    /*
     *  Returns a potential message associated with a tab id (infobar)
     */
    return {
        getMessageForTabId: function (tabId)  {
            return getDataIfAnyFor(tabId).message;
        }
    };
})();

