!------------------------------------------------------------------------
! Copyright 2008, CERFACS, Toulouse, France.
! All rights reserved.
!------------------------------------------------------------------------
!BOP
!EOP
!=============================================================================
!
PROGRAM model1
  !
  ! !USES:
  !
  USE netcdf
  !
  USE mod_kinds_model
  USE mod_prism_proto
  USE mod_prism_def_partition_proto
  USE mod_prism_put_proto
  USE mod_prism_get_proto
  USE mod_prism_grids_writing
  !
  IMPLICIT NONE
  !
  ! By default OASIS3 exchanges data in double precision
  ! To exchange data in single precision with OASIS3, 
  ! the coupler has to be compiled with CPP key "use_realtype_single" 
  ! but this toy is not adapted to the single precision case.
  !
  INTEGER, PARAMETER :: wp = SELECTED_REAL_KIND(12,307) ! double
  !
  !++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !                             DECLARATIONS :
  !++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !
  CHARACTER(len=30), PARAMETER   :: data_filename='grid_model1.nc'
  CHARACTER(len=6)   :: comp_name = 'toyocn'   ! Init Component 6 characters
                                               ! same as in the namcouple
  CHARACTER(len=128) :: comp_out ! name of the output log file 
  CHARACTER(len=3)   :: chout
  !
  ! Grid parameters : 
  INTEGER :: nlon, nlat     ! global dimensions in the (i,j) space
  INTEGER :: nbr_corners_ij ! number of corners in the (i,j) plan
  !
  INTEGER :: nloci, nlocj   ! local dimension of the fields on this pe in the (i,j) space
  INTEGER :: loc_ideb, loc_ifin, loc_jdeb, loc_jfin ! local indexes for the grid
  !
  DOUBLE PRECISION, DIMENSION(:,:), POINTER    :: globalgrid_lon,globalgrid_lat ! lon, lat of the points
  DOUBLE PRECISION, DIMENSION(:,:,:), POINTER  :: globalgrid_clo,globalgrid_cla ! lon, lat of the corners
  INTEGER, DIMENSION(:,:), POINTER             :: indice_mask ! mask defined with OASIS4 convention
                                                              ! 1 == valid point
                                                              ! 0 == masked point
  !
  INTEGER :: mype, npes ! rank and  number of pe
  INTEGER :: localComm  ! local MPI communicator and Initialized
  INTEGER :: comp_id    ! component identification
  !
  INTEGER :: ierror, rank, w_unit
  INTEGER :: i, j, ij
  !
  ! Names of exchanged Fields
  CHARACTER(len=8), PARAMETER      :: var_name1 = 'FSENDOCN' ! 8 characters, field sent by model1
  CHARACTER(len=8), PARAMETER      :: var_name2 = 'FRECVOCN' ! 8 characters, field received by model1
  CHARACTER(len=8), PARAMETER      :: var_name3 = 'FOCNWRIT' ! 8 characters, field written in a file
  !
  ! Used in prism_def_var and prism_def_var_proto
  INTEGER                       :: var_id(3) 
  INTEGER                       :: var_nodims(2) 
  INTEGER                       :: var_type
  !
  !
  REAL (kind=wp), PARAMETER     :: field_ini = -1. ! initialisation of received fields
  !
  INTEGER               ::  ib
  INTEGER, PARAMETER    ::  il_nb_time_steps = 6 ! number of time steps
  INTEGER, PARAMETER    ::  delta_t = 3600       ! time step
  !
  ! Centers arrays of the local grid
  ! used to calculate the field field1_send sent by the model by OASIS3 
  DOUBLE PRECISION, POINTER :: localgrid_lon (:,:)
  DOUBLE PRECISION, POINTER :: localgrid_lat (:,:)
  !
  !
  INTEGER                       :: il_flag          ! Flag for grid writing by proc 0
  !
  INTEGER                       :: itap_sec ! Time used in prism_put_proto and prism_get_proto
  !
  ! Grid partition
  INTEGER                       :: part_id  ! use to connect the partition to the variables
                                            ! in prism_def_var_proto
  INTEGER                       :: var_shape_oasis3(4) ! local dimensions of the arrays to the pe
                                                       ! 2 x field rank (= 4 because fields are of rank = 2)
  INTEGER, POINTER              :: il_paral(:)     ! Definition of process partition
  INTEGER                       :: dim_paral       ! dimension of il_paral, linked to the decomposition
  !
  !
  ! Exchanged local fields arrays
  ! used in routines prism_put_proto and prism_get_proto
  REAL (kind=wp),   POINTER     :: field1_send(:,:)
  REAL (kind=wp),   POINTER     :: field2_recv(:,:)
  REAL (kind=wp),   POINTER     :: field3(:,:)
  !
  !
  !++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !   INITIALISATION 
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !
  !!!!!!!!!!!!!!!!! PRISM_INIT !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  !
  CALL prism_init_comp_proto (comp_id, comp_name, ierror )
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'prism_init_comp_proto', 'Pb in model1')
  !
  ! Unit for output messages : one file for each process
  CALL MPI_Comm_Rank ( MPI_COMM_WORLD, rank, ierror )
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'MPI_Comm_Rank', 'Pb in model1')
  !
  !
  w_unit = 100 + rank
  WRITE(chout,'(I3)') w_unit
  comp_out=comp_name//'.out_'//chout
  !
  OPEN(w_unit,file=TRIM(comp_out),form='formatted')
  WRITE (w_unit,*) '-----------------------------------------------------------'
  WRITE (w_unit,*) TRIM(comp_name), ' Running with reals compiled as kind =',wp
  WRITE (w_unit,*) 'I am component ', TRIM(comp_name), ' rank :',rank
  WRITE (w_unit,*) '----------------------------------------------------------'
  CALL flush(w_unit)
  !
  !!!!!!!!!!!!!!!!! PRISM_GET_LOCALCOMM !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  !
  CALL prism_get_localcomm_proto ( localComm, ierror )
  IF (ierror /= 0) CALL prism_abort_proto(comp_id,'prism_get_localcomm_proto','Pb in model1')
  !
  ! Get MPI sizes
  !
  CALL MPI_Comm_Size ( localComm, npes, ierror )
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'MPI_Comm_Size','Pb in model1')
  !
  CALL MPI_Comm_Rank ( localComm, mype, ierror )
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'MPI_Comm_Rank','Pb in model1')
  !
  !
  WRITE(w_unit,*) 'I am the', TRIM(comp_name), ' ', 'comp', comp_id, 'local rank', mype
  CALL flush(w_unit)
  !
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !  GRID DEFINITION 
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !
  ! Reading global grid netcdf file
  !
  ! Reading dimensions of the global grid
  CALL read_dim_model1(nlon,nlat,nbr_corners_ij,data_filename,w_unit)
  !
  !
  ! Allocation
  ALLOCATE(globalgrid_lon(nlon,nlat), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating globalgrid_lon'
  ALLOCATE(globalgrid_lat(nlon,nlat), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating globalgrid_lat'
  ALLOCATE(globalgrid_clo(nlon,nlat,nbr_corners_ij), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating globalgrid_clo'
  ALLOCATE(globalgrid_cla(nlon,nlat,nbr_corners_ij), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating globalgrid_cla'
  ALLOCATE(indice_mask(nlon,nlat), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating indice_mask'
  !
!SV indice_mask en 2D 
  !
  ! Reading of the longitudes, latitudes, longitude and latitudes of the corners, mask of the global grid
  CALL read_grid_model1(nlon,nlat,nbr_corners_ij,data_filename, w_unit, &
                        globalgrid_lon,globalgrid_lat,&
                        globalgrid_clo,globalgrid_cla,&
                        indice_mask)
  !
  !
  ! 1 A - (Global) grid definition for OASIS3
  ! Writing of the file grids.nc and masks.nc by the processor 0 
  IF (mype == 0) THEN
  ! The convention for the mask 0 for valid point and 1 for masked point for OASIS3
      indice_mask(:,:)=1-indice_mask(:,:) 
      !
      ! Half of the line j=148 is masked 
      DO i=92,182
        indice_mask(i,148)=1
      ENDDO
      !
      CALL prism_start_grids_writing(il_flag)
      IF (il_flag) THEN
          CALL prism_write_grid('torc', nlon, nlat, globalgrid_lon, globalgrid_lat)
          CALL prism_write_corner('torc', nlon, nlat, 4, globalgrid_clo, globalgrid_cla)
          CALL prism_write_mask('torc', nlon, nlat, indice_mask)
          CALL prism_terminate_grids_writing()
      ENDIF
  ENDIF
  !
  !
  !!!!!!!!!!!!!!!!!!!!!!! DEFINITION OF THE LOCAL SHAPE OF THE ARRAYS !!!!!!!!!!!!!!!!!!!!!
  !
  ! Allocate the fields send and received by the model
  !
  nloci = nlon  ! no partition in the I direction
  !
  if (mype .lt. (npes-1)) then
     nlocj = (nlat/npes)
  else
     nlocj = nlat-mype*(nlat/npes)
  endif
  !
  ALLOCATE(field1_send(nloci,nlocj), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating field1_send'
  !
  ALLOCATE(field2_recv(nloci,nlocj), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating field2_recv'
  !
  ALLOCATE(field3(nloci,nlocj), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating field3'
  !
  var_shape_oasis3 (1) = 1
  var_shape_oasis3 (2) = nloci
  var_shape_oasis3 (3) = 1
  var_shape_oasis3 (4) = nlocj
  !
#if defined DECOMP_SERIAL || defined DECOMP_APPLE
  dim_paral = 3
#elif defined DECOMP_BOX 
  dim_paral = 5
#endif
  !
  ALLOCATE(il_paral(dim_paral), STAT=ierror )
  IF ( ierror /= 0 ) WRITE(w_unit,*) 'Error allocating il_paral'
  !
  CALL oasis3_decomp (dim_paral, mype, npes, w_unit, nlon, nlat, il_paral)
  !
  ! Definition of the partition of the grid
  CALL prism_def_partition_proto (part_id, il_paral, ierror)
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'prism_def_partition_proto','Pb in model1')
  !
  !
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ! DEFINITION OF THE LOCAL FIELDS  
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !
  !!!!!!!!!!!!!!! !!!!!!!!! PRISM_DEF_VAR !!!!!!!!!!!!!!!!!!!!!!!!!!!
  !
  !  Define transient variables
  !
  var_nodims(1) = 2    ! Rank of the field array is 2
  var_nodims(2) = 1    ! Bundles always 1 for OASIS3
  !
  ! For real and double precision, var_type = PRISM_Real
  var_type = PRISM_Real
  !
  ! Declaration of the field associated with the partition
  CALL prism_def_var_proto (var_id(1),var_name1, part_id, &
     var_nodims, PRISM_Out, var_shape_oasis3, var_type, ierror)
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'prism_def_var_proto', 'Pb in model1')
  !
  CALL prism_def_var_proto (var_id(2),var_name2, part_id, &
     var_nodims, PRISM_In, var_shape_oasis3, var_type, ierror)
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'prism_def_var_proto', 'Pb in model1')
  !
  CALL prism_def_var_proto (var_id(3),var_name3, part_id, &
     var_nodims, PRISM_Out, var_shape_oasis3, var_type, ierror)
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'prism_def_var_proto', 'Pb in model1')
  !
  !
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !         TERMINATION OF DEFINITION PHASE AND SEARCH  
  ! OASIS3 : only the processes involved in the coupling must call prism_enddef_proto; 
  ! here all processes are involved in coupling
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !
  !!!!!!!!!!!!!!!!!! PRISM_ENDDEF !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  !
  ! Collective call to end the definition phase
  !
  CALL prism_enddef_proto ( ierror )
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'prism_enddef_proto', 'Pb in model1')
  !
  !
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ! SEND AND RECEIVE ARRAYS 
  !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  !
  ! Local portion of the grid
  loc_ideb = 1
  loc_ifin = nlon
  loc_jdeb = ((nlat/npes)*mype) + 1
  IF (mype .LT. (npes-1)) THEN
     loc_jfin = (nlat/npes)*(mype+1)
  ELSE
     loc_jfin = nlat 
  ENDIF
  
  !!!!!!!!!!!!!!!!!!!!!!!!PRISM_PUT/PRISM_GET !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! 
  !
  ! Data exchange 
  !
  ! Time loop
  DO ib=1, il_nb_time_steps
    !
    itap_sec = delta_t * (ib-1) ! Time
    !
    CALL function_sent(nloci, nlocj, &
                       globalgrid_lon(loc_ideb:loc_ifin,loc_jdeb:loc_jfin), &
                       globalgrid_lat(loc_ideb:loc_ifin,loc_jdeb:loc_jfin), &
                       field1_send,ib)
    !
    ! Send FSENDOCN  to model2
    CALL prism_put_proto(var_id(1),itap_sec, field1_send, ierror)
    IF ( ierror .NE. PRISM_Ok .AND. ierror .LT. PRISM_Sent) &
    CALL prism_abort_proto(comp_id, 'prism_put_proto', 'Pb in model1')
    !
    ! Get FRECVOCN from model2
    field2_recv=field_ini
    CALL prism_get_proto(var_id(2),itap_sec, field2_recv, ierror)
    IF ( ierror .NE. PRISM_Ok .AND. ierror .LT. PRISM_Recvd) &
    CALL prism_abort_proto(comp_id, 'prism_get_proto', 'Pb in model1')
    !
    field3(:,:) = field1_send(:,:) 
    !
    ! Write FOCNWRIT in a file
    CALL prism_put_proto(var_id(3),itap_sec, field3, ierror)
    IF ( ierror .NE. PRISM_Ok .AND. ierror .LT. PRISM_Sent) &
    CALL prism_abort_proto(comp_id, 'prism_put_proto', 'Pb in model1')
    !
  ENDDO
  !
  CLOSE (w_unit)
  !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! Collective call to terminate the coupling exchanges
  !
  CALL prism_terminate_proto (ierror)
  IF (ierror /= 0) CALL prism_abort_proto(comp_id, 'prism_terminate_proto', 'Pb in model1')
  !
END PROGRAM MODEL1
!
