/* error.hh - error and warning handling
 * Copyright 2004 Bas Wijnen <wijnen@debian.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SHEVEK_ERROR_HH
#define SHEVEK_ERROR_HH

#include "iostring.hh"
#include <errno.h>
#include <string.h>
#include <glibmm.h>

namespace shevek
{
  /// Set whether shevek_error will throw or not.
  /** Note that setting this to false may trigger bugs, as it is not tested much.
   */
  bool fatal_errors (bool fatal = true);
  /// Internal function for printing the errors.
  void _error_impl (Glib::ustring const &msg, bool is_error, char const *file, unsigned line, char const *fun);
  /// Give an error message with file, line and function information.
  #define shevek_error(x) \
  shevek::_error_impl ((x), true, __FILE__, __LINE__, __FUNCTION__)
  /// Give a warning message with file, line and function information.
  #define shevek_warning(x) \
  shevek::_error_impl ((x), false, __FILE__, __LINE__, __FUNCTION__)
  /// Give an error message with file, line and function information.
  /** Use errno to give extra information about the error.
   */
  #define shevek_error_errno(x) \
  shevek::_error_impl (shevek::ostring ("%s: %s", Glib::ustring (x), strerror (errno)), true, __FILE__, __LINE__, __FUNCTION__)
  /// Give a warning message with file, line and function information.
  /** Use errno to give extra information about the error.
   */
  #define shevek_warning_errno(x) \
  shevek::_error_impl (shevek::ostring ("%s: %s", Glib::ustring (x), strerror (errno)), false, __FILE__, __LINE__, __FUNCTION__)
}

#endif
