/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* file:     parametric_1d.c                                                */
/*                                                                          */
/*                                                                          */
/* description: Support for parametric elements in 1D                       */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by D. Koester (2005)                                                */
/*--------------------------------------------------------------------------*/

static const REAL_B mid_lambda_1d    = { 0.5, 0.5 };
static const REAL_B vertex_bary_1d[] = { { 1.0, 0.0 }, { 0.0, 1.0 } };

/*--------------------------------------------------------------------------*/
/* Functions for affine elements as parametric elements. (suffix 1_1d)      */
/*--------------------------------------------------------------------------*/

static void det1_1d(const EL_INFO *el_info, const QUAD *quad, int N,
		    const REAL lambda[][N_LAMBDA], REAL dets[])
{
  REAL_D *local_coords = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data)->local_coords;
  REAL        det;
  int         n;

  det = DIST_DOW(local_coords[0], local_coords[1]);

  if (quad) N = quad->n_points;

  for (n = 0; n < N; n++)
    dets[n] = det;

  return;
}


static void grd_lambda1_1d(const EL_INFO *el_info, const QUAD *quad,
			int N, const REAL lambda[][N_LAMBDA],
			REAL_D grd_lam[][N_LAMBDA], REAL dets[])
{
  /* FUNCNAME("grd_lambda1_1d"); */
  REAL_D *local_coords = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data)->local_coords;
  int         i, j, n;
  REAL        det2;
  REAL_D      grd_lambda[N_LAMBDA];

  det2 = DST2_DOW(local_coords[0], local_coords[1]);

  for (i = 0; i < DIM_OF_WORLD; i++) {
    grd_lambda[0][i] = (local_coords[0][i] - local_coords[1][i]) / det2;
    grd_lambda[1][i] = - grd_lambda[0][i];
  }

  if (quad) N = quad->n_points;

  dets[0] = sqrt(det2);

  for (n = 0; n < N; n++) {
    for (i = 0; i <= 1; i++)
      for (j = 0; j < DIM_OF_WORLD; j++)
	grd_lam[n][i][j] = grd_lambda[i][j];

    dets[n] = dets[0];
  }

  return;
}

/****************************************************************************/
/* fill_coords1_1d(data): initialize the DOF_REAL_D_VEC coords containing   */
/* the position data of the parametric elements (coordinates of vertices in */
/* this case).                                                              */
/****************************************************************************/

static void fill_coords1_1d(LAGRANGE_PARAM_DATA *data)
{
  DOF_REAL_D_VEC  *coords = data->coords;
  DOF_UCHAR_VEC   *touched_coords = data->touched_coords;
  NODE_PROJECTION *n_proj = data->n_proj, *active_proj;
  TRAVERSE_STACK  *stack = get_traverse_stack();
  const EL_INFO   *el_info;
  FLAGS            fill_flag = CALL_LEAF_EL|FILL_COORDS|FILL_PROJECTION;
  const DOF     *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;
  int              i, n;

  get_dof = coords->fe_space->bas_fcts->get_dof_indices;
  admin   = coords->fe_space->admin;

  el_info = traverse_first(stack, coords->fe_space->mesh, -1, fill_flag);
  while (el_info) {
    const DOF *dof = (*get_dof)(el_info->el, admin, nil);
    REAL      *vec;

    for (i = 0; i < N_VERTICES_1D; i++) {
      vec = coords->vec[dof[i]];
      for (n = 0; n < DIM_OF_WORLD; n++)
	vec[n] = el_info->coord[i][n];

      /* Look for a projection function that applies to vertex[i]. */
      /* Apply this projection if found.                           */

      if(touched_coords)
	touched_coords->vec[dof[i]] = 0;

      if(n_proj && (n_proj->func)) {
	active_proj = el_info->projections[0];

	if(active_proj == n_proj) {
	  n_proj->func(vec, el_info, vertex_bary_1d[i]);

	  if(touched_coords)
	    touched_coords->vec[dof[i]] = 1;
	}
      }
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);
  
  return;
}


/****************************************************************************/
/* refine_interpol1_1d(drdv,list,n): update coords vector during refinement.*/
/****************************************************************************/

static void refine_interpol1_1d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list, int n)
{
  FUNCNAME("refine_interpol1_1d");
  EL      *el;
  REAL_D  *vec = nil;
  DOF     dof_new, dof0, dof1;
  NODE_PROJECTION *n_proj = 
    ((LAGRANGE_PARAM_DATA *)drdv->fe_space->mesh->parametric->data)->n_proj;
  int     n0, j;

  GET_DOF_VEC(vec, drdv);
  n0 = drdv->fe_space->admin->n0_dof[VERTEX];
  el = list->el_info.el;

  dof0 = el->dof[0][n0];           /* left point of refinement elem. */
  dof1 = el->dof[1][n0];           /* right point */
  dof_new = el->child[0]->dof[1][n0];

  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[dof_new][j] = 0.5*(vec[dof0][j] + vec[dof1][j]);

  if((list->el_info.active_projection == n_proj) && n_proj->func)
    (n_proj->func)(vec[dof_new], &list->el_info, mid_lambda_1d);

  return;
}


/*--------------------------------------------------------------------------*/
/* Common functions for higher order elements. (suffix y_1d)                */
/*--------------------------------------------------------------------------*/

#define N_BAS4_1D  5

typedef struct DD_data_1d DD_DATA_1D;
struct DD_data_1d
{
  const QUAD *quad;

  int         n_bas_fcts;
  REAL      (*DD)[N_BAS4_1D];

  DD_DATA_1D *next;
};

static DD_DATA_1D *init_dd_data_1d(const QUAD *quad, const BAS_FCTS *bas_fcts)
{
  FUNCNAME("init_dd_data_1d");
  static DD_DATA_1D *first_dd_data = nil;
  DD_DATA_1D        *data;
  int                iq, i;
  const REAL        *grd;

  DEBUG_TEST_EXIT(bas_fcts->n_bas_fcts <= N_BAS4_1D,
	      "Sorry, only up to 5 local DOFs at the moment.\n");

  for (data = first_dd_data; data; data = data->next)
    if (data->quad == quad) break;

  if (data) return(data);

  data = MEM_ALLOC(1, DD_DATA_1D);

  data->quad       = quad;
  data->n_bas_fcts = bas_fcts->n_bas_fcts;
  data->next       = first_dd_data;

  first_dd_data = data;

  data->DD = 
    (REAL (*)[N_BAS4_1D])alberta_alloc(quad->n_points*sizeof(REAL [N_BAS4_1D]),
				       funcName, __FILE__,  __LINE__);

  for (iq = 0; iq < quad->n_points; iq++) {
    for (i = 0; i < bas_fcts->n_bas_fcts; i++) {
      grd = (*bas_fcts->grd_phi[i])(quad->lambda[iq]);

      data->DD[iq][i] = grd[1] - grd[0];
    }
  }

  return(data);
}

/*--------------------------------------------------------------------------*/
/*---  compute D^t and D^t D, return det(D^t D)                          ---*/
/*--------------------------------------------------------------------------*/

static REAL Dt_and_DtD_1d(REAL_D *F, REAL *DD, int n_bas, REAL_D Dt)
{
  FUNCNAME("Dt_and_DtD_1d");
  int   i, n;
  REAL  val;

  for (n = 0; n < DIM_OF_WORLD; n++) {
    for (Dt[n] = i = 0; i < n_bas; i++)
      Dt[n] += F[i][n]*DD[i];
  }

  val = NRM2_DOW(Dt);

  if (val < 0.0) {
    WARNING("val = %e\n", val);

    for (i = 0; i < n_bas; i++)
      PRINT_REAL_VEC("F", F[i], DIM_OF_WORLD);
    WAIT_REALLY;
  }

  return(val);
}

static void dety_1d(const EL_INFO *el_info, const QUAD *quad, int N,
		    const REAL lambda[][N_LAMBDA], REAL dets[])
{
  /* FUNCNAME("dety_1d"); */
  REAL_D                    Dt;
  int                       iq;
  const BAS_FCTS           *bas_fcts;
  LAGRANGE_PARAM_DATA      *data = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data);
  REAL_D                   *local_coords = data->local_coords;

  /* First of all, check if we are on a parametric simplex.            */
  /* If not, treat this simplex as an affine simplex, even though some */
  /* higher order vertices might be shifted.                           */
  if(data->i_am_affine) {
    det1_1d(el_info, quad, N, lambda, dets);
    return;
  }

  if (quad) {
    static DD_DATA_1D *dd_data = nil;

    if (!dd_data  ||  dd_data->quad != quad) {
      bas_fcts = ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data)
	->coords->fe_space->bas_fcts;

      dd_data = init_dd_data_1d(quad, bas_fcts);
    }

    for (iq = 0; iq < quad->n_points; iq++)
      dets[iq] = 
	sqrt(Dt_and_DtD_1d(local_coords, dd_data->DD[iq],
			   dd_data->n_bas_fcts, Dt));
  }
  else {
    REAL  DD[N_BAS4_1D];
    const REAL *grd;
    int   i, n_bas;

    bas_fcts = ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data)
      ->coords->fe_space->bas_fcts;
    n_bas = bas_fcts->n_bas_fcts;

    for (iq = 0; iq < N; iq++) {
      for (i = 0; i < n_bas; i++) {
	grd = (*bas_fcts->grd_phi[i])(lambda[iq]);

	DD[i] = grd[1] - grd[0];
      }

      dets[iq] = sqrt(Dt_and_DtD_1d(local_coords, DD, n_bas, Dt));
    }
  }

  return;
}

static REAL Lambda_iq_1d(REAL_D *const F, REAL *DD, int n_bas,
			 REAL_D Lambda[N_LAMBDA])
{
  REAL    Dt[DIM_OF_WORLD];
  REAL    DFS_1[DIM_OF_WORLD];
  REAL    det, det_1;
  int     n;

  det = Dt_and_DtD_1d(F, DD, n_bas, Dt);
  det_1 = 1.0/det;

/*--------------------------------------------------------------------------*/
/*--- compute DF_S^{-1} = D^t D^{-1} D^t                                 ---*/
/*--------------------------------------------------------------------------*/
  
  for (n = 0; n < DIM_OF_WORLD; n++)
    DFS_1[n] = det_1*Dt[n];

/*--------------------------------------------------------------------------*/
/*--- finally, \Lambda = \hat\Lambda DF_S^{-1}                           ---*/
/*--------------------------------------------------------------------------*/

  for (n = 0; n < DIM_OF_WORLD; n++) {
    Lambda[0][n] = -DFS_1[n];
    Lambda[1][n] = DFS_1[n];
  }

  return(sqrt(det));
}

static void grd_lambday_1d(const EL_INFO *el_info, const QUAD *quad,
			   int N, const REAL lambda[][N_LAMBDA],
			   REAL_D Lambda[][N_LAMBDA], REAL dets[])
{
  /* FUNCNAME("grd_lambday"); */
  int                       iq;
  const BAS_FCTS           *bas_fcts;
  LAGRANGE_PARAM_DATA      *data = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data);
  REAL_D                   *local_coords = data->local_coords;

  /* First of all, check if we are on a parametric simplex.            */
  /* If not, treat this simplex as an affine simplex, even though some */
  /* higher order vertices might be shifted.                           */
  if(data->i_am_affine) {
    grd_lambda1_1d(el_info,quad, N, lambda, Lambda, dets);
    return;
  }

  if (quad) {
    static DD_DATA_1D *dd_data = nil;

    if (!dd_data  ||  dd_data->quad != quad) {
      bas_fcts = data->coords->fe_space->bas_fcts;

      dd_data = init_dd_data_1d(quad, bas_fcts);
    }

    for (iq = 0; iq < quad->n_points; iq++)
      dets[iq] = Lambda_iq_1d(local_coords, dd_data->DD[iq],
			   dd_data->n_bas_fcts, Lambda[iq]);
  }
  else
  {
    REAL  DD[N_BAS4_1D];
    const REAL *grd;
    int   i, n_bas;

    bas_fcts = data->coords->fe_space->bas_fcts;
    n_bas = bas_fcts->n_bas_fcts;

    for (iq = 0; iq < N; iq++) {
      for (i = 0; i < n_bas; i++) {
	grd = (*bas_fcts->grd_phi[i])(lambda[iq]);

	DD[i] = grd[1] - grd[0];
      }
      dets[iq] = Lambda_iq_1d(local_coords, DD, n_bas, Lambda[iq]);
    }
  }

  return;
}

/*--------------------------------------------------------------------------*/
/* Functions for quadratic elements. (suffix 2_1d)                          */
/*--------------------------------------------------------------------------*/

static void refine_interpol2_1d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				int n_neigh)
{
  /* FUNCNAME("refine_interpol2_1d"); */
  EL                  *el;
  const DOF           *cdof;
  DOF                  pdof;
  const REAL_D        *pvec;
  int                  n, not_all;
  const BAS_FCTS      *bas_fcts = drdv->fe_space->bas_fcts;
  const DOF_ADMIN     *admin = drdv->fe_space->admin;
  NODE_PROJECTION     *n_proj;
  LAGRANGE_PARAM_DATA *data = 
    (LAGRANGE_PARAM_DATA *)list->el_info.mesh->parametric->data;
  DOF_UCHAR_VEC       *touched_coords = data->touched_coords;
  REAL_D              *vec = drdv->vec;
  static const REAL_B child_mid_lambda[] = {
    { 0.75, 0.25 },
    { 0.25, 0.75 }
  };

  el      = list->el_info.el;
  pvec    = bas_fcts->get_real_d_vec(el, drdv, nil);
  pdof    = el->dof[drdv->fe_space->mesh->node[CENTER]]
           [drdv->fe_space->admin->n0_dof[CENTER]];
  n_proj  = data->n_proj;
  not_all = drdv->fe_space->mesh->parametric->not_all;

/****************************************************************************/
/*  DOFs on first child                                                     */
/****************************************************************************/
  cdof = bas_fcts->get_dof_indices(el->child[0], admin, nil);

  if(not_all < 2)
    for (n = 0; n < DIM_OF_WORLD; n++) {
      vec[cdof[1]][n] = pvec[2][n];
      vec[cdof[2]][n] =(0.375*pvec[0][n] - 0.125*pvec[1][n] + 0.75*pvec[2][n]);
    }
  else
    for (n = 0; n < DIM_OF_WORLD; n++) {
      vec[cdof[1]][n] = pvec[2][n];
      vec[cdof[2]][n] = 0.5 * (pvec[0][n] + pvec[2][n]);
    }    

  if(touched_coords) {
    touched_coords->vec[cdof[1]] = touched_coords->vec[pdof];
    touched_coords->vec[cdof[2]] = 0;
  }

  if(n_proj && n_proj->func && (list->el_info.active_projection == n_proj)) {
    (n_proj->func)(vec[cdof[2]], &list->el_info, child_mid_lambda[0]);

    if(touched_coords)
      touched_coords->vec[cdof[2]] = 1;
  }

/****************************************************************************/
/*  DOF on second child                                                     */
/****************************************************************************/

  cdof = (*bas_fcts->get_dof_indices)(el->child[1], admin, nil);

  if(not_all < 2)
    for (n = 0; n < DIM_OF_WORLD; n++)
      vec[cdof[2]][n] =(-0.125*pvec[0][n]+ 0.375*pvec[1][n] + 0.75*pvec[2][n]);
  else 
    for (n = 0; n < DIM_OF_WORLD; n++)
      vec[cdof[2]][n] = 0.5 * (pvec[1][n]+ pvec[2][n]);

  if(touched_coords)
    touched_coords->vec[cdof[2]] = 0;

  if(n_proj && n_proj->func && (list->el_info.active_projection == n_proj)) {
    (n_proj->func)(vec[cdof[2]], &list->el_info, child_mid_lambda[1]);

    if(touched_coords)
      touched_coords->vec[cdof[2]] = 1;
  }
  return;
}

static void coarse_interpol2_1d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list, int n)
{
  FUNCNAME("coarse_interpol2_1d");
  EL                  *el;
  REAL_D              *vec = nil;
  int                  node_v, node_c, n0_v, n0_c, j;
  DOF                  cdof, pdof;
  LAGRANGE_PARAM_DATA *data = 
    (LAGRANGE_PARAM_DATA *)list->el_info.mesh->parametric->data;
  DOF_UCHAR_VEC       *touched_coords = data->touched_coords;

  if (n < 1) return;

  GET_DOF_VEC(vec, drdv);
  el = list->el_info.el;

  node_v = drdv->fe_space->mesh->node[VERTEX];        
  node_c = drdv->fe_space->mesh->node[CENTER]; 
  n0_v = drdv->fe_space->admin->n0_dof[VERTEX];
  n0_c = drdv->fe_space->admin->n0_dof[CENTER];

/****************************************************************************/
/*  copy values at refinement vertex to the parent center DOF.              */
/****************************************************************************/

  cdof = el->child[0]->dof[node_v+1][n0_v];      /* newest vertex is dim */
  pdof = el->dof[node_c][n0_c];

  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[pdof][j] = vec[cdof][j];

  if(touched_coords)
    touched_coords->vec[pdof] = touched_coords->vec[cdof];

  return;
}

static void fill_coords2_1d(LAGRANGE_PARAM_DATA *data)
{
  DOF_REAL_D_VEC  *coords = data->coords;
  DOF_UCHAR_VEC   *touched_coords = data->touched_coords;
  NODE_PROJECTION *n_proj = data->n_proj, *active_proj;
  TRAVERSE_STACK  *stack = get_traverse_stack();
  const EL_INFO   *el_info;
  FLAGS            fill_flag = CALL_LEAF_EL|FILL_COORDS|FILL_PROJECTION;
  const DOF     *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;
  int              i, n;

  get_dof = coords->fe_space->bas_fcts->get_dof_indices;
  admin   = coords->fe_space->admin;

  el_info = traverse_first(stack, coords->fe_space->mesh, -1, fill_flag);
  while (el_info) {
    const DOF *dof = (*get_dof)(el_info->el, admin, nil);
    REAL      *vec;

    /* Fill the two vertices. */
    for (i = 0; i < N_VERTICES_1D; i++) {
      vec = coords->vec[dof[i]];

      for (n = 0; n < DIM_OF_WORLD; n++)
	vec[n] = el_info->coord[i][n];

      /* Look for a projection function that applies to vertex[i]. */
      /* Apply this projection if found.                           */

      if(touched_coords)
	touched_coords->vec[dof[i]] = 0;

      if(n_proj && (n_proj->func)) {
	active_proj = el_info->active_projection;

	if(active_proj == n_proj) {
	  (n_proj->func)(vec, el_info, vertex_bary_1d[i]);

	  if(touched_coords)
	    touched_coords->vec[dof[i]] = 1;
	}
      }
    }

    /* Fill the midpoint. */
    vec = coords->vec[dof[2]];
    for (n = 0; n < DIM_OF_WORLD; n++)
      vec[n] = 0.5 * (el_info->coord[0][n] + el_info->coord[1][n]);

    if(touched_coords)
      touched_coords->vec[dof[2]] = 0;

    if(n_proj && (n_proj->func)) {
      active_proj = el_info->active_projection;
      
      if(active_proj == n_proj) {
	(n_proj->func)(vec, el_info, mid_lambda_1d);

	if(touched_coords)
	  touched_coords->vec[dof[2]] = 1;
      }
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);

  return;
}

static PARAMETRIC lagrange_parametric1_1d = 
  {"1D Lagrange parametric elements of degree 1",
   param_init_element,
   param_coord_to_world,
   nil,
   det1_1d,
   grd_lambda1_1d,
   false, false, nil};

static PARAMETRIC lagrange_parametric2_1d = 
  {"1D Lagrange parametric elements of degree 2",
   param_init_element,
   param_coord_to_world,
   nil,
   dety_1d,
   grd_lambday_1d,
   false, false, nil};
