#!/usr/bin/env lua

local lpeg = require "lpeg"

local BEGIN_COMMENT = lpeg.P("/*")
local END_COMMENT = lpeg.P("*/")
local NOT_BEGIN = (1 - BEGIN_COMMENT)^0
local NOT_END = (1 - END_COMMENT)^0
local FULL_COMMENT_CONTENTS = BEGIN_COMMENT * NOT_END * END_COMMENT
local find_comments = ((1 - lpeg.P"/")^0 * (lpeg.C(FULL_COMMENT_CONTENTS) + 1))^0

local source = io.stdin:read "*a"
local doc_comments = { }
local current_section
local sections = 0
local contents = { }
local results = { }
local header = ""

local output = { }

local function o(fmt, ...)
   output[#output + 1] = fmt:format(...)
end

local function e(fmt, ...)
   io.stderr:write(fmt:format(...))
end

function trim(s)
   return (s:gsub("^%s*(.-)%s*$", "%1"))
end

function add_to_contents(s)
   local c = contents[current_section]
   if not contents[current_section] then
      c = { name = current_section }
      contents[current_section] = c
      sections = sections + 1
      contents[sections] = c
   end

   c[#c + 1] = s
end

function to_anchor(s)
   return (s:gsub("[^a-zA-Z0-9]", "-")):lower()
end

if arg[1] == "-l" then
   for c in source:gmatch "\n%-%-%-([^\n]*)" do
      doc_comments[#doc_comments + 1] = c
   end
else
   for i, v in ipairs { find_comments:match(source) } do
      if v:match("^/**") then
	 local content = v:match "^/%*%*%s*(.*)%s*%*/$"
	 if content then
	    doc_comments[#doc_comments + 1] =
	       content:gsub("\n%s*%*%s+", "\n"):gsub("\n%s*%*%s+", "\n\n"):gsub("%\\t", "\t") 
	 end
      end
   end
end

for _, doc in ipairs(doc_comments) do
   if doc:match "^%%" then
      -- a simple mapping, where the first line is the C prototype,
      -- and any remaining lines are return styles.
      local r = { type = "simple", name = doc:match "^%% ([_a-zA-Z]+)" }
      local ln = 0
      for line in doc:gmatch "[^\r\n]+" do
	 ln = ln + 1
	 if ln == 2 then
	    r.c = trim(line)
	 elseif ln > 2 then
	    local trimmed = trim(line)
	    if #trimmed > 0 then
	       r.l = r.l or { }
	       r.l[#r.l + 1] = trimmed
	    end
	 end
      end
      if not r.c or not r.l then
	 error(r.name .. " has not enough stuff.")
      end
      results[#results + 1] = r
      add_to_contents(r.name)
   elseif doc:match "^#" then
      -- Section header
      current_section = trim(doc:match "# ([^*]+)")
      contents[current_section] = { name = current_section }
      contents[#contents + 1] = contents[current_section]
      results[#results + 1] = {
	 type = "section",
	 name = current_section
      }
   elseif doc:match "^>" then
      -- verbatim output.  If there is anything on the same line as the >
      -- then it is used as a section title for contents purposes.
      local title = doc:match "^>[ ]*([^\n]*)"
      if title and #title > 0 then
	 results[#results + 1] = {
	    type = "verbatim",
	    name = title,
	    content = doc:match "\n(.*)$"
	 }
	 add_to_contents(title)
      else
	 results[#results + 1] = {
	    type = "verbatim",
	    content = doc:match "\n(.*)$"
	 }
      end
   elseif doc:match "^!" then
      -- Emitted verbatim before contents
      header = doc:match "^!(.*)"
   end
end

o(header)

o "## Contents"

for secnum, seccont in ipairs(contents) do
   o("%d. [%s](#%s)", secnum, seccont.name, to_anchor(seccont.name))
   for funcnum, funcname in ipairs(seccont) do
      o("\t%d. [%s](#%s)", funcnum, funcname, to_anchor(funcname))
   end
end


for i, v in ipairs(results) do
   if v.type == "simple" then
      o("\n### %s()", v.name)
      o("C style: `%s`\n", v.c)

      if #v.l < 2 then
	 o("Lua style: `%s`", v.l[1])
      else
	 o("Lua styles:\n")
	 for i, v in pairs(v.l) do
	    o(" * `%s`", v)
	 end
      end
   elseif v.type == "section" then
      o("\n## %s", v.name)
   elseif v.type == "verbatim" then
      if v.name then
	 o("\n### %s", v.name)
      end
      o(v.content)
   end
   o ""
end

io.stdout:write(table.concat(output, "\n"))