package latexDraw.generators.svg;

import latexDraw.figures.Ellipse;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGEllipseElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for an ellipse.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LEllipseSVGGenerator extends LShapeSVGGenerator
{
	public LEllipseSVGGenerator(Ellipse f)
	{
		super(f);
	}

	
	
	/**
	 * Creates an ellipse from an SVG ellipse element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LEllipseSVGGenerator(SVGEllipseElement elt)
	{
		this(new Ellipse(true));
		
		Ellipse e = (Ellipse)getShape();
		
		setSVGParameters(elt);
		e.setFirstPoint(elt.getCx()-elt.getRx(), elt.getCy()-elt.getRy());
		e.setLastPoint(elt.getCx()+elt.getRx(), elt.getCy()+elt.getRy());
		e.setBordersPosition(PSTricksConstants.BORDERS_MIDDLE);
		applyTransformations(elt);
	}
	
	
	
	public LEllipseSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates an ellipse from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LEllipseSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Ellipse(false));

		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		Ellipse e = (Ellipse)getShape();
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGEllipseElement))
			throw new IllegalArgumentException();
		
		SVGEllipseElement main = (SVGEllipseElement)elt2;
		
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		
		double gap = getPositionGap()/2.;
        
		e.setFirstPoint(main.getCx()-main.getRx()+gap, main.getCy()-main.getRy()+gap);
		e.setLastPoint(main.getCx()+main.getRx()-gap, main.getCy()+main.getRy()-gap);
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	

	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null || doc.getFirstChild().getDefs()==null)
			throw new IllegalArgumentException();

		LaTeXDrawPoint2D nw = shape.getTheNWPoint();
		LaTeXDrawPoint2D se = shape.getTheSEPoint();
		SVGElement elt;
		SVGElement root = new SVGGElement(doc);
        root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_ELLIPSE);
        root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
        double gap = getPositionGap();

        if(shape.hasShadow())
        {
        	elt = new SVGEllipseElement((se.x+nw.x)/2., (se.y+nw.y)/2., Math.max(1, (se.x-nw.x+gap)/2.), 
        																 Math.max(1, (se.y-nw.y+gap)/2.), doc);
        	setSVGShadowAttributes(elt, true);
        	root.appendChild(elt);
        }
        
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE))
        {// The background of the borders must be filled is there is a shadow.
        	elt = new SVGEllipseElement((se.x+nw.x)/2., (se.y+nw.y)/2., Math.max(1, (se.x-nw.x+gap)/2.), 
																		Math.max(1, (se.y-nw.y+gap)/2.), doc);
        	setSVGBorderBackground(elt, root);
        }
        
        elt = new SVGEllipseElement((se.x+nw.x)/2., (se.y+nw.y)/2., Math.max(1, (se.x-nw.x+gap)/2.), 
        															Math.max(1, (se.y-nw.y+gap)/2.), doc);
        setSVGAttributes(doc, elt, true);
        root.appendChild(elt);
        
        if(shape.hasDoubleBoundary())
        {
        	elt = new SVGEllipseElement((se.x+nw.x)/2., (se.y+nw.y)/2., Math.max(1, (se.x-nw.x+gap)/2.), 
        																	Math.max(1, (se.y-nw.y+gap)/2.), doc);
        	setSVGDoubleBordersAttributes(elt);
        	root.appendChild(elt);
        }

        setSVGRotationAttribute(root);
        
		return root;
	}
}
