﻿/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 * Author: Martin Hentschel <info@cl-soft.de>
 *
 */

using System.Collections.Generic;
using System.Text.RegularExpressions;
using CCodeGeneration;

namespace LwipSnmpCodeGeneration
{
	public class MibHeaderFile
	{

		#region Fields

		private readonly List<CodeElement> defines = new List<CodeElement>();
		private readonly List<CodeElement> includes = new List<CodeElement>();
		private readonly List<CodeElement> functionDeclarations = new List<CodeElement>();
		private readonly List<CodeElement> variableDeclarations = new List<CodeElement>();

		#endregion

		public MibHeaderFile()
		{
		}

		#region Accessors

		public List<CodeElement> Defines
		{
			get { return this.defines; }
		}

		public List<CodeElement> Includes
		{
			get { return this.includes; }
		}

		public List<CodeElement> FunctionDeclarations
		{
			get { return this.functionDeclarations; }
		}

		public List<CodeElement> VariableDeclarations
		{
			get { return this.variableDeclarations; }
		}

		#endregion

		#region Methods

		public void Save(CGenerator cGenerator)
		{
			CFile cFile = new CFile();

			cFile.AddElement(new Comment("Generated by LwipMibCompiler"));
			cFile.AddElement(EmptyLine.SingleLine);
			
			string headerDefine = cGenerator.FileName;
			headerDefine = new Regex("[^a-zA-Z0-9]").Replace(headerDefine, "_");
			headerDefine = headerDefine.ToUpperInvariant();

			CodeContainerBase e = cFile.AddElement(new PP_Ifdef(headerDefine, inverted: true)) as CodeContainerBase;
			e.AddElement(new PP_Macro(headerDefine, headerDefine));
			e.AddElement(EmptyLine.SingleLine);

			e.AddElement(new PP_Include(LwipDefs.Incl_SnmpOpts));
			e = e.AddElement(new PP_If(LwipDefs.Opt_SnmpEnabled)) as CodeContainerBase;
			e.AddElement(EmptyLine.SingleLine);

			CodeContainerBase cplusplusopen = e.AddElement(new PP_Ifdef("__cplusplus")) as CodeContainerBase;
			cplusplusopen.AddElement(new Code("extern \"C\" {"));
			e.AddElement(EmptyLine.SingleLine);

			if (this.includes.Count > 0)
			{
				e.AddElements(this.includes);
				e.AddElement(EmptyLine.SingleLine);
			}

			if (this.defines.Count > 0)
			{
				e.AddElements(this.defines);
				e.AddElement(EmptyLine.SingleLine);
			}
			
			e.AddElements(this.functionDeclarations, EmptyLine.SingleLine);
			e.AddElements(this.variableDeclarations, EmptyLine.SingleLine);

			e.AddElement(EmptyLine.SingleLine);
			CodeContainerBase cplusplusclose = e.AddElement(new PP_Ifdef("__cplusplus")) as CodeContainerBase;
			cplusplusclose.AddElement(new Code("}"));

			e.AddElement(EmptyLine.SingleLine);
			cFile.Save(cGenerator);
		}

		#endregion
	}
}
