/*
 * Copyright © 2013 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include "piglit-util-gl.h"

/**
 * @file bind_buffer_invalid_index.c
 *
 * Tests that invalid index given to BindBufferRange() or BindBufferBase()
 * is detected and an error is set. The spec says:
 *
 *  "The error INVALID_VALUE is generated by BindBufferRange,
 *   BindBufferOffsetEXT, or BindBufferBase if <target> is
 *   TRANSFORM_FEEDBACK_BUFFER and <index> is greater than or equal to the
 *   value of MAX_TRANSFORM_FEEDBACK_BUFFERS."
 */

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 32;
	config.supports_gl_core_version = 32;
	config.khr_no_error_support = PIGLIT_HAS_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

void
piglit_init(int argc, char **argv)
{
	GLuint xfb;
	GLint max_xfb_buffers;
	bool pass = true;

	piglit_require_extension("GL_ARB_transform_feedback3");

	glGetIntegerv(GL_MAX_TRANSFORM_FEEDBACK_BUFFERS, &max_xfb_buffers);
	if (!piglit_check_gl_error(GL_NO_ERROR)) {
		printf("failed to resolve the maximum number of feedback "
			"buffers\n");
		piglit_report_result(PIGLIT_FAIL);
	}

	if (max_xfb_buffers <= 0) {
		printf("invalid maximum number of feedback buffers: %d\n",
			max_xfb_buffers);
		piglit_report_result(PIGLIT_FAIL);
	}

	/* Set up the transform feedback buffers. */
	glGenBuffers(1, &xfb);

	/* Binding to the maximum supported buffer should be fine. */
	glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, max_xfb_buffers - 1,
			xfb);
	if (!piglit_check_gl_error(GL_NO_ERROR)) {
		printf("failed to bind maximum supported buffer\n");
		pass = false;
	}

	/* Unbind. */
	glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, max_xfb_buffers - 1, 0);

	glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, max_xfb_buffers, xfb);
	if (!piglit_check_gl_error(GL_INVALID_VALUE)) {
		printf("should not be able to bind beyond maximum supported\n");
		pass = false;

		/* Unbind. */
		glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, max_xfb_buffers,
				0);
	}

	/* Binding to the maximum supported buffer should be fine. */
	glBindBufferRange(GL_TRANSFORM_FEEDBACK_BUFFER, max_xfb_buffers - 1,
			xfb, 0, sizeof(GLfloat));
	if (piglit_check_gl_error(GL_NO_ERROR)) {
		/* Unbind. */
		glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER,
			max_xfb_buffers - 1, 0);
	} else {
		printf("failed to bind range in maximum supported buffer\n");
		pass = false;
	}

	glBindBufferRange(GL_TRANSFORM_FEEDBACK_BUFFER, max_xfb_buffers,
			xfb, 0, sizeof(GLfloat));
	if (!piglit_check_gl_error(GL_INVALID_VALUE)) {
		printf("should not be able to bind beyond maximum supported\n");
		pass = false;

		/* Unbind. */
		glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, max_xfb_buffers,
				0);
	}

	glDeleteBuffers(1, &xfb);

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result
piglit_display(void)
{
	/* Should never be reached */
	return PIGLIT_FAIL;
}
