#!/usr/bin/env python

"""
Remove expired events from quota journals.

Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from os.path import abspath, split
import sys

# Find the modules.

try:
    import imiptools
except ImportError:
    parent = abspath(split(split(__file__)[0])[0])
    if split(parent)[1] == "imip-agent":
        sys.path.append(parent)

from codecs import getwriter
from imiptools.config import settings
from imiptools.dates import get_datetime, get_default_timezone, get_time, \
                            to_utc_datetime
from imiptools.stores import get_journal

def remove_expired_entries(entries, expiry):

    "Remove from 'entries' events that end at or before 'expiry'."

    removed = []
    i = 0

    while i < len(entries):
        period = entries[i]

        if period.get_end_point() <= expiry:
            removed.append(period)
            del entries[i]
        else:
            i += 1

    return removed

def update_entries(journal, quota, expiry, store, verbose):

    """
    Using the given 'journal' process quota records for the given 'quota', with
    the given 'expiry' time used to expire events ending before or at this time,
    with None meaning the current time.

    If 'store' is set, the stored details will be updated; otherwise, the
    details will be written to standard output.

    If 'verbose' is set, messages will be written to standard error.
    """

    if not store:
        stdout = getwriter("utf-8")(sys.stdout)
    if verbose:
        stderr = getwriter("utf-8")(sys.stderr)

    if not expiry:
        expiry = get_time()

    journal.acquire_lock(quota)

    try:
        for user in journal.get_quota_users(quota):
            if verbose:
                print >>stderr, user

            entries = journal.get_entries(quota, user)
            removed = remove_expired_entries(entries, expiry)

            if verbose:
                for period in removed:
                    print >>stderr, "\t".join(("Removed",) + period.as_tuple(strings_only=True))

            # Store the processed entries.

            if store:
                journal.set_entries(quota, user, entries)

            # Alternatively, just write the entries to standard output.

            else:
                for period in entries:
                    print >>stdout, "\t".join(period.as_tuple(strings_only=True))
    finally:
        journal.release_lock(quota)

# Main program.

if __name__ == "__main__":

    # Interpret the command line arguments.

    quotas = []
    args = []
    store_type = []
    journal_dir = []
    expiry = []
    ignored = []

    # Collect quota details first, switching to other arguments when encountering
    # switches.

    l = quotas

    for arg in sys.argv[1:]:
        if arg in ("-s", "-v"):
            args.append(arg)
            l = ignored
        elif arg == "-T":
            l = store_type
        elif arg == "-j":
            l = journal_dir
        elif arg == "-e":
            l = expiry
        else:
            l.append(arg)

    try:
        quota = quotas[0]
    except IndexError:
        print >>sys.stderr, """\
Usage: %s <quota> <options>

Need a quota along with the -s option if updating the journal.
Specify -v for additional messages on standard error.

General options:

-e  Indicates an expiry time for events (default is now)
-j  Indicates the journal directory location
-T  Indicates the store type (the configured value if omitted)
""" % split(sys.argv[0])[1]
        sys.exit(1)

    # Define any other options.

    store = "-s" in args
    verbose = "-v" in args

    # Override defaults if indicated.

    getvalue = lambda value, default=None: value and value[0] or default

    store_type = getvalue(store_type, settings["STORE_TYPE"])
    journal_dir = getvalue(journal_dir)
    expiry = getvalue(expiry)

    if expiry:
        expiry = to_utc_datetime(get_datetime(expiry), get_default_timezone())
        if not expiry:
            print >>sys.stderr, "Expiry time must be a valid datetime."
            sys.exit(1)

    # Obtain store-related objects.

    journal = get_journal(store_type, journal_dir)

    # Obtain a list of users for processing.

    if quota in ("*", "all"):
        quotas = journal.get_quotas()

    # Process the given users.

    if verbose:
        stderr = getwriter("utf-8")(sys.stderr)

    for quota in quotas:
        if verbose:
            print >>stderr, quota
        update_entries(journal, quota, expiry, store, verbose)

# vim: tabstop=4 expandtab shiftwidth=4
